package app.raybritton.elog.ui

import android.app.Activity
import android.content.Context
import android.content.Intent
import android.graphics.Color
import android.os.Bundle
import android.text.Editable
import android.text.TextWatcher
import android.view.LayoutInflater
import android.view.MenuItem
import android.view.View
import android.view.ViewGroup
import androidx.recyclerview.widget.DefaultItemAnimator
import androidx.recyclerview.widget.LinearLayoutManager
import androidx.recyclerview.widget.RecyclerView
import app.raybritton.elog.ELog
import app.raybritton.elog.ELogConfig
import app.raybritton.elog.R
import app.raybritton.elog.arch.LogModule
import app.raybritton.elog.data.LogFile
import app.raybritton.elog.data.LogPreferences
import kotlinx.android.synthetic.main.elog_activity_log_list.*
import kotlinx.android.synthetic.main.elog_element_log_file.view.*
import java.text.SimpleDateFormat
import java.util.*
import kotlin.concurrent.thread
import kotlin.math.log

class LogListActivity : Activity() {

    private lateinit var adapter: LogAdapter

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        setContentView(R.layout.elog_activity_log_list)
        elog_list_toolbar.setBackgroundColor(ELogConfig.toolbarBackgroundColor)
        elog_list_toolbar.setTitleTextColor(ELogConfig.toolbarForegroundColor)
        getDrawable(R.drawable.elog_ic_close)?.setTint(ELogConfig.toolbarForegroundColor)
        getDrawable(R.drawable.elog_ic_send)?.setTint(ELogConfig.toolbarForegroundColor)
        setStatusBarColor(ELogConfig.toolbarBackgroundColor)

        setActionBar(elog_list_toolbar)
        actionBar?.setTitle(R.string.elog_list_title)
        actionBar?.setDisplayHomeAsUpEnabled(true)
        actionBar?.setDisplayShowHomeEnabled(true)
        actionBar?.setHomeAsUpIndicator(R.drawable.elog_ic_close)

        var deviceId = LogPreferences.deviceId
        if (deviceId.isEmpty()) {
            deviceId = getString(R.string.elog_no_logs_yet)
        }
        elog_list_deviceid.text = getString(R.string.elog_device_id, deviceId)

        elog_list_content.layoutManager = LinearLayoutManager(this)
        elog_list_content.itemAnimator = DefaultItemAnimator()
        thread {
            val logs = LogModule.logManager.getLogs()
            runOnUiThread {
                adapter = LogAdapter(logs, ::onViewLog, ::onSendLog)
                elog_list_content.adapter = adapter
                elog_list_progress.visibility = View.GONE
            }
        }

        elog_list_search.addTextChangedListener(object: TextWatcher {
            override fun afterTextChanged(editable: Editable) {
                val searchText = editable.toString()
                if (this@LogListActivity::adapter.isInitialized) {
                    adapter.searchText = searchText
                }
            }

            override fun beforeTextChanged(p0: CharSequence?, p1: Int, p2: Int, p3: Int) {

            }

            override fun onTextChanged(p0: CharSequence?, p1: Int, p2: Int, p3: Int) {

            }

        })
    }

    override fun onOptionsItemSelected(item: MenuItem): Boolean {
        if (item.itemId == android.R.id.home) {
            finish()
        }
        return super.onOptionsItemSelected(item)
    }

    private fun onViewLog(logFile: LogFile) {
        LogViewerActivity.start(this, logFile)
    }

    private fun onSendLog(logFile: LogFile) {
        if (logFile.code == null) {
            LogSubmitActivity.start(this, logFile, true)
        } else {
            AlreadySentActivity.start(this, logFile.code)
        }
    }

    private class LogAdapter(
        private val logFiles: List<LogFile>,
        private val onViewLog: (LogFile) -> Unit, private val onSendLog: (LogFile) -> Unit
    ) : RecyclerView.Adapter<LogAdapter.LogFileViewHolder>() {
        private val data = mutableListOf<LogFile>()
        private val dateFormatter = SimpleDateFormat("yyyy/MM/dd HH:mm:ss", Locale.UK)

        init {
            data.addAll(logFiles)
        }

        var searchText = ""
            set(value) {
                field = value
                data.clear()
                if (field == "") {
                    data.addAll(logFiles)
                } else {
                    data.addAll(logFiles.filter { it.code?.contains(field) == true })
                }
                notifyDataSetChanged()
            }

        override fun onCreateViewHolder(parent: ViewGroup, viewType: Int): LogFileViewHolder {
            val view = LayoutInflater.from(parent.context).inflate(R.layout.elog_element_log_file, parent, false)
            return LogFileViewHolder(view)
        }

        override fun getItemCount() = data.size

        override fun onBindViewHolder(holder: LogFileViewHolder, position: Int) {
            val logFile = data[position]
            holder.itemView.elog_log_file_timestamp.text = holder.itemView.elog_log_file_timestamp.context.getString(R.string.elog_session_started, dateFormatter.format(Date(logFile.created)))
            if (logFile.code == null) {
                holder.itemView.elog_log_file_code.setTextColor(Color.BLACK)
                holder.itemView.elog_log_file_code.setText(R.string.elog_not_submitted)
                holder.itemView.elog_log_file_code.paint.isFakeBoldText = false
            } else {
                holder.itemView.elog_log_file_code.setTextColor(holder.getColorById(R.color.elog_code))
                holder.itemView.elog_log_file_code.text = logFile.code
                holder.itemView.elog_log_file_code.paint.isFakeBoldText = true
            }
        }

        private inner class LogFileViewHolder(view: View) : RecyclerView.ViewHolder(view) {
            init {
                view.setOnClickListener {
                    onViewLog(data[adapterPosition])
                }
                view.elog_log_file_send.setOnClickListener {
                    onSendLog(data[adapterPosition])
                }
            }
        }
    }

    companion object {
        internal fun start(ctx: Context) {
            val intent = Intent(ctx, LogListActivity::class.java)
            intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
            ctx.startActivity(intent)
        }
    }
}
