package app.raybritton.elog

import android.annotation.SuppressLint
import android.content.Context
import android.graphics.Color
import android.net.Uri
import app.raybritton.elog.arch.LogModule
import app.raybritton.elog.data.ELogTree
import app.raybritton.elog.data.Result
import app.raybritton.elog.ui.LogListActivity
import app.raybritton.elog.ui.LogSubmitActivity
import okhttp3.Interceptor
import timber.log.Timber
import java.io.File
import kotlin.concurrent.thread
import kotlin.random.Random

@SuppressLint("StaticFieldLeak") //It points to application context
object ELog {
    var idAutoGen = ELogIdGen.NUMBER

    internal var uploadServerUrl = ""

    var contactName = "your manager"
    var contactEmail = "dev@company.com"
    var appPackage: String = ""
    var appBuild: String = ""
    var appVersion: Int = 0
    var appVersionName: String = ""

    //Optional, if set included in the log when submitted
    //This can be changed whenever but will only affect logs
    //submitted after it's changed
    var userId = ""
    //Optional, if set included in the log when submitted
    //This can be changed whenever but will only affect logs
    //submitted after it's changed
    var additionalInfo = ""

    /**
     * Set before logging anything ever, set automatically if empty when first log is written
     */
    var deviceId: String = ""

    fun plant(context: Context, serverUrl: String) {
        LogModule.context = context.applicationContext
        val parsed = Uri.parse(serverUrl)
        if (parsed.toString() != serverUrl) {
            throw IllegalArgumentException("ELog url is invalid")
        }
        uploadServerUrl = serverUrl
        Timber.plant(ELogTree())
        LogModule.logManager.cleanupLogs()
    }

    fun openLogScreen() {
        LogListActivity.start(LogModule.context)
    }

    fun submitCurrentLog(manual: Boolean = false, logTitle: String? = null) {
        thread {
            val log = LogModule.logManager.getCurrentLog()
            if (log == null) {
                Timber.e("No log files!")
            } else {
                LogSubmitActivity.startNewTask(LogModule.context, log, manual, logTitle)
            }
        }
    }

    fun submitCurrentLogSilently(logTitle: String? = null, manual: Boolean = false) {
        thread {
            Thread.sleep(5000) //wait 5 seconds for any extra logs
            // I'm not sure about this, it seems hacky just because it's a sleep
            // but it seems really useful to just get whatever happens next
            // after an error
            val log = LogModule.logManager.getCurrentLog()
            if (log == null) {
                Timber.e("No log files!")
            } else {
                LogModule.logUploader.upload(log, manual, logTitle) {
                    when (it) {
                        is Result.Success -> {
                            Timber.i("Log submitted with code ${it.code}")
                        }
                        is Result.Error -> Timber.e(it.e, "Log failed to submit because ${it.e.message}")
                    }
                }
            }
        }
    }

    internal val logDir by lazy {
        val dir = File(LogModule.context.filesDir, "elogs")
        dir.mkdirs()
        dir
    }

    internal fun generateCode(mode: ELogIdGen = idAutoGen): String {
        when (mode) {
            ELogIdGen.NUMBER -> {
                return Random.nextInt(100000, 999999).toString()
            }
            ELogIdGen.ANIMAL -> {
                val animal = ANIMALS[Random.nextInt(0, ANIMALS.size - 1)]
                val adjective = ADJECTIVES[Random.nextInt(0, ADJECTIVES.size - 1)]

                return "$adjective $animal"
            }
            ELogIdGen.ANIMAL_AND_NUMBER -> {
                val number = Random.nextInt(10, 99)
                val animal = ANIMALS[Random.nextInt(0, ANIMALS.size - 1)]
                val adjective = ADJECTIVES[Random.nextInt(0, ADJECTIVES.size - 1)]

                return "$adjective $animal $number"
            }
        }

    }
}

enum class ELogIdGen {
    NUMBER, ANIMAL, ANIMAL_AND_NUMBER
}

private val ANIMALS = arrayOf(
    "Alpaca",
    "Badger",
    "Crane",
    "Dolphin",
    "Goose",
    "Hawk",
    "Jackal",
    "Koala",
    "Lizard",
    "Monkey",
    "Newt",
    "Owl",
    "Pug",
    "Wolf"
);
private val ADJECTIVES = arrayOf(
    "Intrepid",
    "Speedy",
    "Minty",
    "Chilly",
    "Smooth",
    "Raspy",
    "Giant",
    "Exhausted",
    "Splendid",
    "Helpful",
    "Determined"
);

object ELogConfig {
    /**
     * Seconds
     */
    var connectTimeout = 300L
    /**
     * Seconds
     */
    var readTimeout = 300L
    /**
     * Seconds
     */
    var writeTimeout = 300L

    var interceptor: Interceptor? = null

    /**
     * Logs that have been submitted will be deleted after this much time has passed
     */
    var maxSubmittedLogAgeMs = 2678400000 //One month

    /**
     * Logs that have not been submitted will be deleted after this much time has passed
     */
    var maxUnsubmittedLogAgeMs = 604800000 //One week


    var toolbarBackgroundColor = Color.parseColor("#005cb2")
    var toolbarForegroundColor = Color.parseColor("#eeeeee")

    /**
     * Endpoint to submit logs to
     */
    var apiPath = "/api/log"
}