package app.raybritton.tokenstorage.crypto

import android.annotation.TargetApi
import android.content.Context
import android.os.Build
import android.security.KeyPairGeneratorSpec
import android.security.keystore.KeyGenParameterSpec
import android.security.keystore.KeyProperties
import android.util.Base64
import app.raybritton.tokenstorage.CryptoLogging
import java.io.File
import java.math.BigInteger
import java.security.KeyPairGenerator
import java.security.KeyStore
import java.security.PrivateKey
import java.security.PublicKey
import java.security.spec.AlgorithmParameterSpec
import java.util.Calendar
import java.util.UUID
import javax.crypto.Cipher
import javax.security.auth.x500.X500Principal

/**
 * Uses asymmetric keys (RSA/ECB/PKCS1Padding) to encrypt passphrase used for encryption/decryption (AES/CBC/PKCS5Padding)
 *
 * @param alias The alias for the certificate
 *
 * @param options Options for user auth requirements
 *
 * A salt is generated when using this class, it is saved a file called 'cert-crypto-salt' (by default) in
 * the apps files dir, if this is deleted or changed the stored data will be lost
 *
 * A key is generated when using this class, it is saved a file called 'cert-crypto-key' (by default) in
 * the apps files dir, if this is deleted or changed the stored data will be lost
 */
class CertCrypto(private val context: Context,
                 private val alias: String = context.packageName + "_tokens",
                 saltDir: File = context.filesDir,
                 saltFilename: String = "cert-crypto-salt",
                 private val keyDir: File = context.filesDir,
                 private val keyFilename: String = "cert-crypto-key",
                 private val options: Options = Options.NO_AUTH()) : Crypto, CryptoLogging {
    sealed class Options(internal val userAuth: Boolean, internal val invalidate: Boolean, internal val loginSeconds: Int) {
        /**
         * No keyguard is required
         */
        class NO_AUTH: Options(false, false, 0)

        /**
         * Keyguard is required
         * NOTE: If set to true and the devices keyguard is disabled or reset (by device admin, etc) all data is lost
         *
         * @param invalidateOnChange If true, when the biometrics on the device change (adding or removing fingerprints, etc) all data is lost
         * Only used on Android N+
         *
         * If the user is not authenticated and any method is called a UserNotAuthenticatedException is thrown
         */
        class KEYGUARD_REQUIRED(invalidateOnChange: Boolean): Options(true, invalidateOnChange, 0)

        /**
         * Keyguard is required and user must have authenticated within the last maxSecondsSinceUnlock seconds
         * NOTE: If set to true and the devices keyguard is disabled or reset (by device admin, etc) all data is lost
         *
         * @param invalidateOnChange If true, when the biometrics on the device change (adding or removing fingerprints, etc) all data is lost
         * Only used on Android N+
         *
         * @param maxSecondsSinceUnlock Maximum number of seconds that is allowed to have passed since the user last authenticated.
         *
         * If the time has past or the user is not authenticated and any method is called a UserNotAuthenticatedException is thrown
         */
        class LOGGED_IN(invalidateOnChange: Boolean, maxSecondsSinceUnlock: Int): Options(true, invalidateOnChange, maxSecondsSinceUnlock)
    }

    private val KEYSTORE_TYPE = "AndroidKeyStore"
    private val KEYSTORE_ALGORITM = "RSA"
    private val ENCRYPT_ALGORITHM = "RSA/ECB/PKCS1Padding"

    private var privateKey: PrivateKey? = null
    private var publicKey: PublicKey? = null

    private val internalCrypto = PassphraseCrypto(saltDir, saltFilename)

    init {
        fine("init with ($alias, $saltDir, $saltFilename, $keyDir, $keyFilename, $options)")
    }

    private val passphrase by lazy {
        fine("passphrase creation")
        certVerify()
        val passphraseFile = File(keyDir, keyFilename)
        if (passphraseFile.exists()) {
            certDecrypt(passphraseFile.readLines()[0])
        } else {
            val newPassphrase = UUID.randomUUID().toString() + UUID.randomUUID().toString() + UUID.randomUUID().toString() + UUID.randomUUID().toString()
            passphraseFile.writeText(certEncrypt(newPassphrase))
            newPassphrase
        }
    }

    private fun loadCert(): Boolean {
        fine("loadCert()")
        val keyStore = KeyStore.getInstance(KEYSTORE_TYPE)
        keyStore.load(null)
        if (keyStore.containsAlias(alias)) {
            privateKey = keyStore.getKey(alias, null) as PrivateKey
            publicKey = keyStore.getCertificate(alias).publicKey
            return true
        } else {
            return false
        }
    }

    override fun encrypt(plaintext: String): String {
        fine("encrypt($plaintext)")
        return internalCrypto.encrypt(plaintext)
    }
    override fun decrypt(encrypted: String): String {
        fine("decrypt($encrypted)")
        return internalCrypto.decrypt(encrypted)
    }

    private fun certEncrypt(plaintext: String): String {
        fine("certEncrypt($plaintext)")
        val cipher = Cipher.getInstance(ENCRYPT_ALGORITHM)
        cipher.init(Cipher.ENCRYPT_MODE, publicKey)

        val encrypted = cipher.doFinal(plaintext.toByteArray())
        return Base64.encodeToString(encrypted, Base64.NO_WRAP)
    }

    private fun certDecrypt(encrypted: String): String {
        fine("certDecrypt($encrypted)")
        val cipher = Cipher.getInstance(ENCRYPT_ALGORITHM)
        cipher.init(Cipher.DECRYPT_MODE, privateKey)

        val encryptedData = Base64.decode(encrypted, Base64.DEFAULT)
        val decodedData = cipher.doFinal(encryptedData)
        return String(decodedData)
    }

    private fun certVerify() {
        fine("certVerify()")
        if (!loadCert()) {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                setupPassphrase(::marshmallowKeyGen)
            } else {
                setupPassphrase(::preMarshmallowKeyGen)
            }
        }
    }

    override fun verify() {
        fine("verify()")
        internalCrypto.passphrase = passphrase
        internalCrypto.verify()
    }

    override fun reset() {
        fine("reset()")
        File(keyDir, keyFilename).delete()
        internalCrypto.reset()
    }

    private fun setupPassphrase(gen: () -> AlgorithmParameterSpec) {
        fine("setupPassphrase()")
        val keyPairGenerator = KeyPairGenerator.getInstance(
                KEYSTORE_ALGORITM, KEYSTORE_TYPE)
        keyPairGenerator.initialize(gen())
        val keyPair = keyPairGenerator.generateKeyPair()
        privateKey = keyPair.private
        publicKey = keyPair.public
        debug("Public key: $publicKey")
    }

    @TargetApi(Build.VERSION_CODES.M)
    private fun marshmallowKeyGen(): KeyGenParameterSpec {
        fine("marshmallowKeyGen()")
        return KeyGenParameterSpec.Builder(
                alias,
                KeyProperties.PURPOSE_ENCRYPT or KeyProperties.PURPOSE_DECRYPT)
                .setBlockModes(KeyProperties.BLOCK_MODE_ECB)
                .setEncryptionPaddings(KeyProperties.ENCRYPTION_PADDING_RSA_PKCS1)
                .setUserAuthenticationRequired(options.userAuth)
                .setUserAuthenticationValidityDurationSeconds(options.loginSeconds)
                .setupBiometrics()
                .build()
    }

    @Suppress("DEPRECATION")
    private fun preMarshmallowKeyGen(): KeyPairGeneratorSpec {
        fine("marshmallowKeyGen()")
        return KeyPairGeneratorSpec.Builder(context)
                .setAlias(alias)
                .setSerialNumber(BigInteger.valueOf(32945367343536L))
                .setSubject(X500Principal("CN=$alias Certificate, O=${context.packageName}"))
                .setStartDate(Calendar.getInstance().also { it.add(Calendar.YEAR, -1) }.time)
                .setEndDate(Calendar.getInstance().also { it.add(Calendar.YEAR, 30) }.time)
                .build()
    }

    private fun KeyGenParameterSpec.Builder.setupBiometrics(): KeyGenParameterSpec.Builder {
        fine("setupBiometrics()")
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            this.setInvalidatedByBiometricEnrollment(options.invalidate)
        }
        return this
    }

    companion object {
        /**
         * For Java developers (as the default options will be fine for nearly every app)
         */
        @JvmStatic
        fun defaults(context: Context): CertCrypto {
            return CertCrypto(context,
                    context.packageName + "_tokens",
                    context.filesDir,
                    "cert-crypto-salt",
                    context.filesDir,
                    "cert-crypto-key",
                    Options.NO_AUTH())
        }
    }
}