package app.raybritton.tokenstorage

import app.raybritton.tokenstorage.crypto.Crypto
import app.raybritton.tokenstorage.keyCrypto.KeyCrypto
import app.raybritton.tokenstorage.persistence.Persistence

/**
 * Use to secure store strings
 */
class TokenStorage<C : Crypto, P : Persistence, KC : KeyCrypto>(
        val crypto: C,
        val persistence: P,
        val keyCrypto: KC): CryptoLogging {

    init {
        debug("init with (${crypto.javaClass.simpleName}, ${persistence.javaClass.simpleName}, ${keyCrypto.javaClass.simpleName})")
    }

    /**
     * Remove all strings and ids
     */
    fun clearAll() {
        fine("clearAll()")
        debug("clearing ${persistence.keys().size} entries")
        crypto.verify()
        keyCrypto.verify()
        persistence.clearAll()
    }

    /**
     * Remove the string with the matching key (if present)
     */
    fun clear(key: String) {
        fine("clear($key)")
        debug("clearing ${persistence.keys().size} entries")
        crypto.verify()
        keyCrypto.verify()
        persistence.clear(keyCrypto.encrypt(key))
    }

    /**
     * Save the key and value
     */
    fun save(key: String, plaintext: String) {
        fine("save($key, $plaintext)")
        crypto.verify()
        keyCrypto.verify()
        persistence.save(keyCrypto.encrypt(key), crypto.encrypt(plaintext))
    }

    /**
     * Load the string matching the key, null if none found
     */
    fun load(key: String): String? {
        fine("load($key)")
        crypto.verify()
        keyCrypto.verify()
        val encrypted = persistence.load(keyCrypto.encrypt(key))
        if (encrypted == null) {
            debug("no match")
            return null
        } else {
            debug("found")
            return crypto.decrypt(encrypted)
        }
    }

    /**
     * Get all the keys stored
     */
    fun keys(): List<String> {
        fine("keys()")
        crypto.verify()
        keyCrypto.verify()
        return persistence.keys().map { keyCrypto.decrypt(it) }
    }

    /**
     * True if the key has a value in storage
     */
    fun contains(key: String): Boolean {
        fine("contains($key)")
        crypto.verify()
        keyCrypto.verify()
        return persistence.contains(keyCrypto.encrypt(key))
    }

    /**
     * Returns an object representing a key with save and load methods just for that key
     */
    fun wrap(key: String): TokenWrapper {
        fine("wrap($key)")
        return TokenWrapper(this, key)
    }

    /**
     * Removes all files, etc generated by any crypto or persistence where possible
     * It also invalidates this instance of TokenStorage, continuing to use it will cause
     * undefined behaviour
     */
    fun reset() {
        fine("reset()")
        crypto.reset()
        persistence.reset()
    }


}