package app.raybritton.tokenstorage

import app.raybritton.tokenstorage.crypto.Crypto
import app.raybritton.tokenstorage.keyCrypto.KeyCrypto
import app.raybritton.tokenstorage.persistence.Persistence

/**
 * Use to secure store strings
 */
open class TokenStorage<C : Crypto, P : Persistence, KC : KeyCrypto>(
        val crypto: C,
        val persistence: P,
        val keyCrypto: KC) {

    /**
     * Remove all strings and ids
     */
    fun clearAll() {
        crypto.verify()
        keyCrypto.verify()
        persistence.clearAll()
    }

    /**
     * Remove the string with the matching key (if present)
     */
    fun clear(key: String) {
        crypto.verify()
        keyCrypto.verify()
        persistence.clear(keyCrypto.encrypt(key))
    }

    /**
     * Save the key and value
     */
    fun save(key: String, plaintext: String) {
        crypto.verify()
        keyCrypto.verify()
        persistence.save(keyCrypto.encrypt(key), crypto.encrypt(plaintext))
    }

    /**
     * Load the string matching the key, null if none found
     */
    fun load(key: String): String? {
        crypto.verify()
        keyCrypto.verify()
        val encrypted = persistence.load(keyCrypto.encrypt(key))
        if (encrypted == null) {
            return null
        } else {
            return crypto.decrypt(encrypted)
        }
    }

    /**
     * Get all the keys stored
     */
    fun keys(): List<String> {
        crypto.verify()
        keyCrypto.verify()
        return persistence.keys().map { keyCrypto.decrypt(it) }
    }

    /**
     * True if the key has a value in storage
     */
    fun contains(key: String): Boolean {
        crypto.verify()
        keyCrypto.verify()
        return persistence.contains(keyCrypto.encrypt(key))
    }

    /**
     * Returns an object representing a key with save and load methods just for that key
     */
    fun wrap(key: String): TokenWrapper {
        return TokenWrapper(this, key)
    }
}