package app.raybritton.tokenstorage.persistence

import java.util.concurrent.Executors

/**
 * read() and write() should be implemented as blocking, this class will manage threading
 */
abstract class NoRandomAccessPersistence : Persistence {
    private val threadPool = Executors.newCachedThreadPool()
    private lateinit var data : MutableMap<String, String>

    override fun clearAll() {
        verify()
        data.clear()
        writeToDisk()
    }

    override fun clear(key: String) {
        verify()
        data.remove(key)
        writeToDisk()
    }

    override fun save(key: String, encrypted: String) {
        verify()
        data[key] = encrypted
        writeToDisk()
    }

    override fun load(key: String): String? {
        verify()
        return data[key]
    }

    override fun keys(): List<String> {
        verify()
        return data.keys.toList()
    }

    override fun contains(key: String): Boolean {
        return data.contains(key)
    }

    private fun verify() {
        if (!this::data.isInitialized) {
            data = read().toMutableMap()
        }
    }

    private fun writeToDisk() {
        threadPool.submit {
            write(data)
        }
    }

    /**
     * This will be called from a thread (from a thread pool managed by NoRandomAccessPersistence)
     */
    abstract fun write(data: Map<String, String>)

    /**
     * This will be run on the callers thread
     */
    abstract fun read(): Map<String, String>
}
