package app.raybritton.tokenstorage.keyCrypto

import android.util.Base64
import java.io.File
import java.util.UUID
import javax.crypto.Cipher
import javax.crypto.SecretKey
import javax.crypto.SecretKeyFactory
import javax.crypto.spec.PBEKeySpec
import javax.crypto.spec.PBEParameterSpec


/**
 * Uses a passphrase (string) to encrypt and decrypt the strings, for example this could be
 * a PIN entered by the user.
 *
 * the encrypt() method will always produce the same encrypted string for a plaintext string
 *
 * Important:
 * passphrase MUST be set before calling any method
 *
 * A salt is generated when using this class, it is saved a file called 'key-crypto-salt' (by default) in
 * the apps files dir, if this is deleted or changed the stored data will be lost
 */
class PassphraseKeyCrypto(private val saltDir: File,
                          private val saltFilename: String = "key-crypto-salt") : KeyCrypto {
    private val ALGORITHM = "PBEWithMD5AndDES"

    private var key: SecretKey? = null
    private var cipher: Cipher? = null

    /**
     * This can be set at any point, any number of times
     * When changed the crypto is rebuilt, if the passphrase does not match the
     * passphrase used to encrypt the data it will be mangled when decrypting but not lost
     */
    var passphrase: String? = null
        set(value) {
            field = value
            invalidate()
        }

    private val salt by lazy {
        val saltFile = File(saltDir, saltFilename)
        if (saltFile.exists()) {
            saltFile.readLines()[0]
        } else {
            val newSalt = UUID.randomUUID().toString()
            saltFile.writeText(newSalt)
            newSalt
        }
    }

    private fun invalidate() {
        if (passphrase == null) {
            key = null
            cipher = null
        } else {
            val keyFactory = SecretKeyFactory.getInstance(ALGORITHM)
            key = keyFactory.generateSecret(PBEKeySpec(passphrase!!.toCharArray()))
            cipher = Cipher.getInstance(ALGORITHM)
        }
    }

    override fun encrypt(plaintext: String): String {
        val bytes = plaintext.toByteArray()
        cipher!!.init(Cipher.ENCRYPT_MODE, key, PBEParameterSpec(salt.toByteArray(), 20))
        return String(Base64.encode(cipher!!.doFinal(bytes), Base64.NO_WRAP))
    }

    override fun decrypt(encrypted: String): String {
        val bytes = Base64.decode(encrypted, Base64.NO_WRAP)
        cipher!!.init(Cipher.DECRYPT_MODE, key, PBEParameterSpec(salt.toByteArray(), 20))
        return String(cipher!!.doFinal(bytes))
    }

    override fun verify() {
        if (passphrase == null || key == null) {
            throw IllegalStateException("passphrase must be set before use")
        }
        if (passphrase.isNullOrEmpty()) {
            throw IllegalStateException("passphrase must have content")
        }
    }
}