/*
 * Decompiled with CFR 0.152.
 */
package androidx.test.espresso.base;

import android.os.Handler;
import android.os.Looper;
import android.os.Message;
import android.util.Log;
import androidx.annotation.NonNull;
import androidx.test.espresso.IdlingPolicies;
import androidx.test.espresso.IdlingPolicy;
import androidx.test.espresso.IdlingResource;
import androidx.test.espresso.base.IdleNotifier;
import androidx.test.espresso.base.LooperIdlingResourceInterrogationHandler;
import androidx.test.espresso.util.TracingUtil;
import androidx.test.internal.util.Checks;
import androidx.test.platform.tracing.Tracer;
import androidx.test.platform.tracing.Tracing;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.concurrent.Callable;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.FutureTask;
import javax.inject.Inject;
import javax.inject.Singleton;

@Singleton
public final class IdlingResourceRegistry {
    private static final String TAG = IdlingResourceRegistry.class.getSimpleName();
    private static final int DYNAMIC_RESOURCE_HAS_IDLED = 1;
    private static final int TIMEOUT_OCCURRED = 2;
    private static final int IDLE_WARNING_REACHED = 3;
    private static final int POSSIBLE_RACE_CONDITION_DETECTED = 4;
    private static final Object TIMEOUT_MESSAGE_TAG = new Object();
    private static final IdleNotificationCallback NO_OP_CALLBACK = new IdleNotificationCallback(){

        @Override
        public void allResourcesIdle() {
        }

        @Override
        public void resourcesStillBusyWarning(List<String> busys) {
        }

        @Override
        public void resourcesHaveTimedOut(List<String> busys) {
        }
    };
    private final List<IdlingState> idlingStates = new ArrayList<IdlingState>();
    private final Looper looper;
    private final Handler handler;
    private final Dispatcher dispatcher;
    private final Tracing tracer;
    private IdleNotificationCallback idleNotificationCallback = NO_OP_CALLBACK;

    @Inject
    IdlingResourceRegistry(Looper looper, @NonNull Tracing tracer) {
        this.looper = looper;
        this.tracer = tracer;
        this.dispatcher = new Dispatcher();
        this.handler = new Handler(looper, (Handler.Callback)this.dispatcher);
    }

    public IdlingResourceRegistry(Looper looper) {
        this(looper, Tracing.getInstance());
    }

    public void sync(final Iterable<IdlingResource> resources, final Iterable<Looper> loopers) {
        if (Looper.myLooper() != this.looper) {
            this.runSynchronouslyOnMainThread(new Callable<Void>(){

                @Override
                public Void call() {
                    IdlingResourceRegistry.this.sync(resources, loopers);
                    return null;
                }
            });
        } else {
            HashMap<String, IdlingResource> resourcesToRegister = new HashMap<String, IdlingResource>();
            for (IdlingResource resource : resources) {
                if (resourcesToRegister.containsKey(resource.getName())) {
                    this.logDuplicateRegistrationError(resource, (IdlingResource)resourcesToRegister.get(resource.getName()));
                    continue;
                }
                resourcesToRegister.put(resource.getName(), resource);
            }
            for (Looper looper : loopers) {
                LooperIdlingResourceInterrogationHandler resource = LooperIdlingResourceInterrogationHandler.forLooper(looper);
                if (resourcesToRegister.containsKey(resource.getName())) {
                    this.logDuplicateRegistrationError(resource, (IdlingResource)resourcesToRegister.get(resource.getName()));
                    continue;
                }
                resourcesToRegister.put(resource.getName(), resource);
            }
            ArrayList<IdlingResource> resourcesToUnRegister = new ArrayList<IdlingResource>();
            for (IdlingState oldState : this.idlingStates) {
                IdlingResource ir = (IdlingResource)resourcesToRegister.remove(oldState.resource.getName());
                if (null == ir) {
                    resourcesToUnRegister.add(oldState.resource);
                    continue;
                }
                if (oldState.resource == ir) continue;
                resourcesToUnRegister.add(oldState.resource);
                resourcesToRegister.put(ir.getName(), ir);
            }
            this.unregisterResources(resourcesToUnRegister);
            this.registerResources(new ArrayList(resourcesToRegister.values()));
        }
    }

    public boolean registerResources(final List<? extends IdlingResource> resourceList) {
        if (Looper.myLooper() != this.looper) {
            return this.runSynchronouslyOnMainThread(new Callable<Boolean>(){

                @Override
                public Boolean call() {
                    return IdlingResourceRegistry.this.registerResources(resourceList);
                }
            });
        }
        boolean allRegisteredSuccessfully = true;
        for (IdlingResource idlingResource : resourceList) {
            Checks.checkNotNull((Object)idlingResource.getName(), (Object)"IdlingResource.getName() should not be null");
            boolean duplicate = false;
            for (IdlingState oldState : this.idlingStates) {
                if (!idlingResource.getName().equals(oldState.resource.getName())) continue;
                this.logDuplicateRegistrationError(idlingResource, oldState.resource);
                duplicate = true;
                break;
            }
            if (!duplicate) {
                IdlingState is = new IdlingState(idlingResource, this.handler);
                this.idlingStates.add(is);
                is.registerSelf();
                continue;
            }
            allRegisteredSuccessfully = false;
        }
        return allRegisteredSuccessfully;
    }

    public boolean unregisterResources(final List<? extends IdlingResource> resourceList) {
        if (Looper.myLooper() != this.looper) {
            return this.runSynchronouslyOnMainThread(new Callable<Boolean>(){

                @Override
                public Boolean call() {
                    return IdlingResourceRegistry.this.unregisterResources(resourceList);
                }
            });
        }
        boolean allUnregisteredSuccessfully = true;
        for (IdlingResource idlingResource : resourceList) {
            boolean found = false;
            for (int i = 0; i < this.idlingStates.size(); ++i) {
                if (!this.idlingStates.get((int)i).resource.getName().equals(idlingResource.getName())) continue;
                this.idlingStates.get(i).closeSpan();
                this.idlingStates.remove(i);
                found = true;
                break;
            }
            if (found) continue;
            allUnregisteredSuccessfully = false;
            Log.e((String)TAG, (String)String.format(Locale.ROOT, "Attempted to unregister resource that is not registered: '%s'. Resource list: %s", idlingResource.getName(), this.getResources()));
        }
        return allUnregisteredSuccessfully;
    }

    public void registerLooper(Looper looper, boolean considerWaitIdle) {
        Checks.checkNotNull((Object)looper);
        Checks.checkArgument((Looper.getMainLooper() != looper ? 1 : 0) != 0, (Object)"Not intended for use with main looper!");
        this.registerResources(Collections.singletonList(LooperIdlingResourceInterrogationHandler.forLooper(looper)));
    }

    public List<IdlingResource> getResources() {
        if (Looper.myLooper() != this.looper) {
            return this.runSynchronouslyOnMainThread(new Callable<List<IdlingResource>>(){

                @Override
                public List<IdlingResource> call() {
                    return IdlingResourceRegistry.this.getResources();
                }
            });
        }
        ArrayList<IdlingResource> irs = new ArrayList<IdlingResource>();
        for (IdlingState is : this.idlingStates) {
            irs.add(is.resource);
        }
        return irs;
    }

    boolean allResourcesAreIdle() {
        Checks.checkState((Looper.myLooper() == this.looper ? 1 : 0) != 0);
        for (IdlingState is : this.idlingStates) {
            if (is.idle) {
                is.setIdle(is.resource.isIdleNow());
            }
            if (is.idle) continue;
            return false;
        }
        if (Log.isLoggable((String)TAG, (int)3)) {
            Log.d((String)TAG, (String)"All idling resources are idle.");
        }
        return true;
    }

    void notifyWhenAllResourcesAreIdle(IdleNotificationCallback callback) {
        Checks.checkNotNull((Object)callback);
        Checks.checkState((Looper.myLooper() == this.looper ? 1 : 0) != 0);
        Checks.checkState((this.idleNotificationCallback == NO_OP_CALLBACK ? 1 : 0) != 0, (Object)"Callback has already been registered.");
        if (this.allResourcesAreIdle()) {
            callback.allResourcesIdle();
        } else {
            this.idleNotificationCallback = callback;
            this.scheduleTimeoutMessages();
        }
    }

    IdleNotifier<IdleNotificationCallback> asIdleNotifier() {
        return new IdleNotifier<IdleNotificationCallback>(){

            @Override
            public boolean isIdleNow() {
                return IdlingResourceRegistry.this.allResourcesAreIdle();
            }

            @Override
            public void cancelCallback() {
                IdlingResourceRegistry.this.cancelIdleMonitor();
            }

            @Override
            public void registerNotificationCallback(IdleNotificationCallback cb) {
                IdlingResourceRegistry.this.notifyWhenAllResourcesAreIdle(cb);
            }
        };
    }

    void cancelIdleMonitor() {
        this.dispatcher.deregister();
    }

    private <T> T runSynchronouslyOnMainThread(Callable<T> task) {
        FutureTask<T> futureTask = new FutureTask<T>(task);
        this.handler.post(futureTask);
        try {
            return futureTask.get();
        }
        catch (CancellationException ce) {
            throw new RuntimeException(ce);
        }
        catch (ExecutionException ee) {
            throw new RuntimeException(ee);
        }
        catch (InterruptedException ie) {
            throw new RuntimeException(ie);
        }
    }

    private void scheduleTimeoutMessages() {
        IdlingPolicy warning = IdlingPolicies.getDynamicIdlingResourceWarningPolicy();
        Message timeoutWarning = this.handler.obtainMessage(3, TIMEOUT_MESSAGE_TAG);
        this.handler.sendMessageDelayed(timeoutWarning, warning.getIdleTimeoutUnit().toMillis(warning.getIdleTimeout()));
        Message timeoutError = this.handler.obtainMessage(2, TIMEOUT_MESSAGE_TAG);
        IdlingPolicy error = IdlingPolicies.getDynamicIdlingResourceErrorPolicy();
        this.handler.sendMessageDelayed(timeoutError, error.getIdleTimeoutUnit().toMillis(error.getIdleTimeout()));
    }

    List<String> getBusyResources() {
        ArrayList<String> busyResourceNames = new ArrayList<String>();
        ArrayList<IdlingState> racyResources = new ArrayList<IdlingState>();
        for (IdlingState state : this.idlingStates) {
            if (state.idle) continue;
            if (state.resource.isIdleNow()) {
                racyResources.add(state);
                continue;
            }
            busyResourceNames.add(state.resource.getName());
        }
        if (!racyResources.isEmpty()) {
            Message raceBuster = this.handler.obtainMessage(4, TIMEOUT_MESSAGE_TAG);
            raceBuster.obj = racyResources;
            this.handler.sendMessage(raceBuster);
            return null;
        }
        return busyResourceNames;
    }

    private void logDuplicateRegistrationError(IdlingResource newResource, IdlingResource oldResource) {
        Log.e((String)TAG, (String)String.format(Locale.ROOT, "Attempted to register resource with same names: %s. R1: %s R2: %s.\nDuplicate resource registration will be ignored.", newResource.getName(), newResource, oldResource));
    }

    private class Dispatcher
    implements Handler.Callback {
        private Dispatcher() {
        }

        public boolean handleMessage(Message m) {
            switch (m.what) {
                case 1: {
                    this.handleResourceIdled(m);
                    break;
                }
                case 3: {
                    this.handleTimeoutWarning();
                    break;
                }
                case 2: {
                    this.handleTimeout();
                    break;
                }
                case 4: {
                    this.handleRaceCondition(m);
                    break;
                }
                default: {
                    Log.w((String)TAG, (String)("Unknown message type: " + m));
                    return false;
                }
            }
            return true;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void handleResourceIdled(Message m) {
            IdlingState is = (IdlingState)m.obj;
            is.setIdle(true);
            boolean unknownResource = true;
            boolean allIdle = true;
            for (IdlingState state : IdlingResourceRegistry.this.idlingStates) {
                boolean bl = allIdle = allIdle && state.idle;
                if (!unknownResource && !allIdle) break;
                if (!unknownResource || state != is) continue;
                unknownResource = false;
            }
            if (unknownResource) {
                Log.i((String)TAG, (String)("Ignoring message from unregistered resource: " + is.resource));
                return;
            }
            if (allIdle) {
                try {
                    IdlingResourceRegistry.this.idleNotificationCallback.allResourcesIdle();
                }
                finally {
                    this.deregister();
                }
            }
        }

        private void handleTimeoutWarning() {
            List<String> busyResources = IdlingResourceRegistry.this.getBusyResources();
            if (busyResources == null) {
                IdlingResourceRegistry.this.handler.sendMessage(IdlingResourceRegistry.this.handler.obtainMessage(3, TIMEOUT_MESSAGE_TAG));
            } else {
                IdlingPolicy warning = IdlingPolicies.getDynamicIdlingResourceWarningPolicy();
                IdlingResourceRegistry.this.idleNotificationCallback.resourcesStillBusyWarning(busyResources);
                IdlingResourceRegistry.this.handler.sendMessageDelayed(IdlingResourceRegistry.this.handler.obtainMessage(3, TIMEOUT_MESSAGE_TAG), warning.getIdleTimeoutUnit().toMillis(warning.getIdleTimeout()));
            }
        }

        private void handleTimeout() {
            List<String> busyResources = IdlingResourceRegistry.this.getBusyResources();
            if (busyResources == null) {
                IdlingResourceRegistry.this.handler.sendMessage(IdlingResourceRegistry.this.handler.obtainMessage(2, TIMEOUT_MESSAGE_TAG));
            } else {
                try {
                    IdlingResourceRegistry.this.idleNotificationCallback.resourcesHaveTimedOut(busyResources);
                }
                finally {
                    this.deregister();
                }
            }
        }

        private void handleRaceCondition(Message m) {
            for (IdlingState is : (List)m.obj) {
                if (is.idle) continue;
                throw new IllegalStateException(String.format(Locale.ROOT, "Resource %s isIdleNow() is returning true, but a message indicating that the resource has transitioned from busy to idle was never sent.", is.resource.getName()));
            }
        }

        private void deregister() {
            IdlingResourceRegistry.this.handler.removeCallbacksAndMessages(TIMEOUT_MESSAGE_TAG);
            IdlingResourceRegistry.this.idleNotificationCallback = NO_OP_CALLBACK;
        }
    }

    private class IdlingState
    implements IdlingResource.ResourceCallback {
        final IdlingResource resource;
        final Handler handler;
        private boolean idle;
        Tracer.Span tracerSpan;

        private IdlingState(IdlingResource resource, Handler handler) {
            this.resource = resource;
            this.handler = handler;
        }

        private void registerSelf() {
            this.resource.registerIdleTransitionCallback((IdlingResource.ResourceCallback)this);
            this.setIdle(this.resource.isIdleNow());
        }

        private void closeSpan() {
            if (this.tracerSpan != null) {
                this.tracerSpan.close();
                this.tracerSpan = null;
                if (!this.idle) {
                    Log.w((String)TAG, (String)("Closing span for resource not idle: " + this.resource.getName()));
                }
            }
        }

        public void setIdle(boolean idle) {
            if (!idle && this.tracerSpan == null) {
                this.tracerSpan = this.createUnmanagedTracerSpan(TracingUtil.getSpanName("IdleResource", this.resource.getName(), new Object[0]));
            } else if (idle && this.tracerSpan != null) {
                this.tracerSpan.close();
                this.tracerSpan = null;
            }
            this.idle = idle;
        }

        private Tracer.Span createUnmanagedTracerSpan(String name) {
            return IdlingResourceRegistry.this.tracer.beginSpan(name);
        }

        public void onTransitionToIdle() {
            Message m = this.handler.obtainMessage(1);
            m.obj = this;
            this.handler.sendMessage(m);
        }
    }

    static interface IdleNotificationCallback {
        public void allResourcesIdle();

        public void resourcesStillBusyWarning(List<String> var1);

        public void resourcesHaveTimedOut(List<String> var1);
    }
}

