/*
 * Decompiled with CFR 0.152.
 */
package androidx.camera.video.internal.audio;

import android.annotation.SuppressLint;
import androidx.annotation.GuardedBy;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.camera.core.Logger;
import androidx.camera.core.impl.utils.executor.CameraXExecutors;
import androidx.camera.video.internal.audio.AudioSettings;
import androidx.camera.video.internal.audio.AudioStream;
import androidx.camera.video.internal.audio.AudioUtils;
import androidx.core.util.Preconditions;
import java.nio.ByteBuffer;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.FutureTask;
import java.util.concurrent.atomic.AtomicBoolean;

public class BufferedAudioStream
implements AudioStream {
    private static final String TAG = "BufferedAudioStream";
    private static final int DEFAULT_BUFFER_SIZE_IN_FRAME = 1024;
    private static final int DEFAULT_QUEUE_SIZE = 500;
    private static final int DATA_WAITING_TIME_MILLIS = 1;
    private final AtomicBoolean mIsStarted = new AtomicBoolean(false);
    private final AtomicBoolean mIsReleased = new AtomicBoolean(false);
    @GuardedBy(value="mLock")
    private final Queue<AudioData> mAudioDataQueue = new ConcurrentLinkedQueue<AudioData>();
    private final Executor mProducerExecutor = CameraXExecutors.newSequentialExecutor((Executor)CameraXExecutors.audioExecutor());
    private final Object mLock = new Object();
    @GuardedBy(value="mLock")
    @Nullable
    private AudioData mAudioDataNotFullyRead = null;
    private final AudioStream mAudioStream;
    private final int mBytesPerFrame;
    private final int mSampleRate;
    private final int mQueueMaxSize;
    private final AtomicBoolean mIsCollectingAudioData = new AtomicBoolean(false);
    private int mBufferSize;

    public BufferedAudioStream(@NonNull AudioStream audioStream, @NonNull AudioSettings audioSettings) {
        this.mAudioStream = audioStream;
        this.mBytesPerFrame = audioSettings.getBytesPerFrame();
        this.mSampleRate = audioSettings.getSampleRate();
        Preconditions.checkArgument(((long)this.mBytesPerFrame > 0L ? 1 : 0) != 0, (Object)"mBytesPerFrame must be greater than 0.");
        Preconditions.checkArgument(((long)this.mSampleRate > 0L ? 1 : 0) != 0, (Object)"mSampleRate must be greater than 0.");
        this.mQueueMaxSize = 500;
        this.mBufferSize = 1024 * this.mBytesPerFrame;
    }

    @Override
    public void start() throws AudioStream.AudioStreamException, IllegalStateException {
        this.checkNotReleasedOrThrow();
        if (this.mIsStarted.getAndSet(true)) {
            return;
        }
        FutureTask<Object> startTask = new FutureTask<Object>(() -> {
            try {
                this.mAudioStream.start();
                this.startCollectingAudioData();
            }
            catch (AudioStream.AudioStreamException e) {
                throw new RuntimeException(e);
            }
        }, null);
        this.mProducerExecutor.execute(startTask);
        try {
            startTask.get();
        }
        catch (InterruptedException | ExecutionException e) {
            this.mIsStarted.set(false);
            throw new AudioStream.AudioStreamException(e);
        }
    }

    @Override
    public void stop() throws IllegalStateException {
        this.checkNotReleasedOrThrow();
        if (!this.mIsStarted.getAndSet(false)) {
            return;
        }
        this.mProducerExecutor.execute(() -> {
            this.mIsCollectingAudioData.set(false);
            this.mAudioStream.stop();
            Object object = this.mLock;
            synchronized (object) {
                this.mAudioDataNotFullyRead = null;
                this.mAudioDataQueue.clear();
            }
        });
    }

    @Override
    public void release() {
        if (this.mIsReleased.getAndSet(true)) {
            return;
        }
        this.mProducerExecutor.execute(() -> {
            this.mIsCollectingAudioData.set(false);
            this.mAudioStream.release();
            Object object = this.mLock;
            synchronized (object) {
                this.mAudioDataNotFullyRead = null;
                this.mAudioDataQueue.clear();
            }
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    @SuppressLint(value={"BanThreadSleep"})
    @NonNull
    public AudioStream.PacketInfo read(@NonNull ByteBuffer byteBuffer) {
        boolean isWaitingForData;
        this.checkNotReleasedOrThrow();
        this.checkStartedOrThrow();
        this.updateCollectionBufferSizeAsync(byteBuffer.remaining());
        AudioStream.PacketInfo packetInfo = AudioStream.PacketInfo.of(0, 0L);
        do {
            Object object = this.mLock;
            synchronized (object) {
                AudioData audioData = this.mAudioDataNotFullyRead;
                this.mAudioDataNotFullyRead = null;
                if (audioData == null) {
                    audioData = this.mAudioDataQueue.poll();
                }
                if (audioData != null) {
                    packetInfo = audioData.read(byteBuffer);
                    if (audioData.getRemainingBufferSizeInBytes() > 0) {
                        this.mAudioDataNotFullyRead = audioData;
                    }
                }
            }
            boolean bl = isWaitingForData = packetInfo.getSizeInBytes() <= 0 && this.mIsStarted.get() && !this.mIsReleased.get();
            if (!isWaitingForData) continue;
            try {
                Thread.sleep(1L);
            }
            catch (InterruptedException e) {
                Logger.w((String)TAG, (String)"Interruption while waiting for audio data", (Throwable)e);
                break;
            }
        } while (isWaitingForData);
        return packetInfo;
    }

    @Override
    public void setCallback(@Nullable AudioStream.AudioStreamCallback callback, @Nullable Executor executor) {
        Preconditions.checkState((!this.mIsStarted.get() ? 1 : 0) != 0, (String)"AudioStream can not be started when setCallback.");
        this.checkNotReleasedOrThrow();
        Preconditions.checkArgument((callback == null || executor != null ? 1 : 0) != 0, (Object)"executor can't be null with non-null callback.");
        this.mProducerExecutor.execute(() -> this.mAudioStream.setCallback(callback, executor));
    }

    private void checkNotReleasedOrThrow() {
        Preconditions.checkState((!this.mIsReleased.get() ? 1 : 0) != 0, (String)"AudioStream has been released.");
    }

    private void checkStartedOrThrow() {
        Preconditions.checkState((boolean)this.mIsStarted.get(), (String)"AudioStream has not been started.");
    }

    private void updateCollectionBufferSizeAsync(int bufferSize) {
        this.mProducerExecutor.execute(() -> this.updateCollectionBufferSize(bufferSize));
    }

    private void updateCollectionBufferSize(int bufferSize) {
        if (this.mBufferSize == bufferSize) {
            return;
        }
        int originalBufferSize = this.mBufferSize;
        int newFrameSize = bufferSize / this.mBytesPerFrame;
        this.mBufferSize = newFrameSize * this.mBytesPerFrame;
        Logger.d((String)TAG, (String)("Update buffer size from " + originalBufferSize + " to " + this.mBufferSize));
    }

    private void startCollectingAudioData() {
        if (this.mIsCollectingAudioData.getAndSet(true)) {
            return;
        }
        this.collectAudioData();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void collectAudioData() {
        if (!this.mIsCollectingAudioData.get()) {
            return;
        }
        ByteBuffer byteBuffer = ByteBuffer.allocateDirect(this.mBufferSize);
        AudioStream.PacketInfo packetInfo = this.mAudioStream.read(byteBuffer);
        AudioData audioData = new AudioData(byteBuffer, packetInfo, this.mBytesPerFrame, this.mSampleRate);
        int queueMaxSize = this.mQueueMaxSize;
        Object object = this.mLock;
        synchronized (object) {
            this.mAudioDataQueue.offer(audioData);
            while (this.mAudioDataQueue.size() > queueMaxSize) {
                this.mAudioDataQueue.poll();
                Logger.w((String)TAG, (String)"Drop audio data due to full of queue.");
            }
        }
        if (this.mIsCollectingAudioData.get()) {
            this.mProducerExecutor.execute(this::collectAudioData);
        }
    }

    private static class AudioData {
        private final int mBytesPerFrame;
        private final int mSampleRate;
        private final ByteBuffer mByteBuffer;
        private long mTimestampNs;

        AudioData(@NonNull ByteBuffer byteBuffer, @NonNull AudioStream.PacketInfo packetInfo, int bytesPerFrame, int sampleRate) {
            byteBuffer.rewind();
            int bufferSize = byteBuffer.limit() - byteBuffer.position();
            if (bufferSize != packetInfo.getSizeInBytes()) {
                throw new IllegalStateException("Byte buffer size is not match with packet info: " + bufferSize + " != " + packetInfo.getSizeInBytes());
            }
            this.mBytesPerFrame = bytesPerFrame;
            this.mSampleRate = sampleRate;
            this.mByteBuffer = byteBuffer;
            this.mTimestampNs = packetInfo.getTimestampNs();
        }

        public int getRemainingBufferSizeInBytes() {
            return this.mByteBuffer.remaining();
        }

        public AudioStream.PacketInfo read(@NonNull ByteBuffer byteBuffer) {
            int readSizeInBytes;
            long timestampNs = this.mTimestampNs;
            int originalSourcePosition = this.mByteBuffer.position();
            int originalDestinationPosition = byteBuffer.position();
            if (this.mByteBuffer.remaining() > byteBuffer.remaining()) {
                readSizeInBytes = byteBuffer.remaining();
                long readFrames = AudioUtils.sizeToFrameCount(readSizeInBytes, this.mBytesPerFrame);
                long readDurationNs = AudioUtils.frameCountToDurationNs(readFrames, this.mSampleRate);
                this.mTimestampNs += readDurationNs;
                ByteBuffer duplicatedByteBuffer = this.mByteBuffer.duplicate();
                duplicatedByteBuffer.position(originalSourcePosition).limit(originalSourcePosition + readSizeInBytes);
                byteBuffer.put(duplicatedByteBuffer).limit(originalDestinationPosition + readSizeInBytes).position(originalDestinationPosition);
            } else {
                readSizeInBytes = this.mByteBuffer.remaining();
                byteBuffer.put(this.mByteBuffer).limit(originalDestinationPosition + readSizeInBytes).position(originalDestinationPosition);
            }
            this.mByteBuffer.position(originalSourcePosition + readSizeInBytes);
            return AudioStream.PacketInfo.of(readSizeInBytes, timestampNs);
        }
    }
}

