/*
 * Decompiled with CFR 0.152.
 */
package androidx.camera.video.internal;

import android.media.MediaCodec;
import android.media.MediaCodecInfo;
import android.media.MediaCodecList;
import android.media.MediaFormat;
import android.os.Build;
import android.text.TextUtils;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.camera.core.Logger;
import androidx.camera.video.internal.compat.Api28Impl;
import androidx.camera.video.internal.compat.Api31Impl;
import androidx.core.util.Preconditions;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Locale;
import java.util.Scanner;
import java.util.concurrent.TimeUnit;

public final class DebugUtils {
    private static final String TAG = "DebugUtils";
    private static final String CODEC_CAPS_PREFIX = "[CodecCaps] ";
    private static final String VIDEO_CAPS_PREFIX = "[VideoCaps] ";
    private static final String AUDIO_CAPS_PREFIX = "[AudioCaps] ";
    private static final String ENCODER_CAPS_PREFIX = "[EncoderCaps] ";

    private DebugUtils() {
    }

    @NonNull
    public static String readableUs(long time) {
        return DebugUtils.readableMs(TimeUnit.MICROSECONDS.toMillis(time));
    }

    @NonNull
    public static String readableMs(long time) {
        return DebugUtils.formatInterval(time);
    }

    @NonNull
    public static String readableBufferInfo(@NonNull MediaCodec.BufferInfo bufferInfo) {
        StringBuilder sb = new StringBuilder();
        sb.append("Dump BufferInfo: " + bufferInfo.toString() + "\n");
        sb.append("\toffset: " + bufferInfo.offset + "\n");
        sb.append("\tsize: " + bufferInfo.size + "\n");
        sb.append("\tflag: " + bufferInfo.flags);
        ArrayList<String> flagList = new ArrayList<String>();
        if ((bufferInfo.flags & 4) != 0) {
            flagList.add("EOS");
        }
        if ((bufferInfo.flags & 2) != 0) {
            flagList.add("CODEC_CONFIG");
        }
        if ((bufferInfo.flags & 1) != 0) {
            flagList.add("KEY_FRAME");
        }
        if ((bufferInfo.flags & 8) != 0) {
            flagList.add("PARTIAL_FRAME");
        }
        if (!flagList.isEmpty()) {
            sb.append(" (").append(TextUtils.join((CharSequence)" | ", flagList)).append(")");
        }
        sb.append("\n");
        sb.append("\tpresentationTime: " + bufferInfo.presentationTimeUs + " (" + DebugUtils.readableUs(bufferInfo.presentationTimeUs) + ")\n");
        return sb.toString();
    }

    private static String formatInterval(long millis) {
        long hr = TimeUnit.MILLISECONDS.toHours(millis);
        long min = TimeUnit.MILLISECONDS.toMinutes(millis - TimeUnit.HOURS.toMillis(hr));
        long sec = TimeUnit.MILLISECONDS.toSeconds(millis - TimeUnit.HOURS.toMillis(hr) - TimeUnit.MINUTES.toMillis(min));
        long ms = millis - TimeUnit.HOURS.toMillis(hr) - TimeUnit.MINUTES.toMillis(min) - TimeUnit.SECONDS.toMillis(sec);
        return String.format(Locale.US, "%02d:%02d:%02d.%03d", hr, min, sec, ms);
    }

    @NonNull
    public static String dumpMediaCodecListForFormat(@NonNull MediaCodecList mediaCodecList, @NonNull MediaFormat mediaFormat) {
        StringBuilder sb = new StringBuilder();
        DebugUtils.logToString(sb, "[Start] Dump MediaCodecList for mediaFormat " + mediaFormat);
        String mime = mediaFormat.getString("mime");
        for (MediaCodecInfo mediaCodecInfo : mediaCodecList.getCodecInfos()) {
            if (!mediaCodecInfo.isEncoder()) continue;
            try {
                Preconditions.checkArgument((mime != null ? 1 : 0) != 0);
                MediaCodecInfo.CodecCapabilities caps = mediaCodecInfo.getCapabilitiesForType(mime);
                Preconditions.checkArgument((caps != null ? 1 : 0) != 0);
                DebugUtils.logToString(sb, "[Start] [" + mediaCodecInfo.getName() + "]");
                DebugUtils.dumpCodecCapabilities(sb, caps, mediaFormat);
                DebugUtils.logToString(sb, "[End] [" + mediaCodecInfo.getName() + "]");
            }
            catch (IllegalArgumentException e) {
                DebugUtils.logToString(sb, "[" + mediaCodecInfo.getName() + "] does not support mime " + mime);
            }
        }
        DebugUtils.logToString(sb, "[End] Dump MediaCodecList");
        String log = sb.toString();
        DebugUtils.stringToLog(log);
        return log;
    }

    @NonNull
    public static String dumpCodecCapabilities(@NonNull String mimeType, @NonNull MediaCodec codec, @NonNull MediaFormat mediaFormat) {
        StringBuilder sb = new StringBuilder();
        try {
            MediaCodecInfo.CodecCapabilities caps = codec.getCodecInfo().getCapabilitiesForType(mimeType);
            Preconditions.checkArgument((caps != null ? 1 : 0) != 0);
            DebugUtils.dumpCodecCapabilities(sb, caps, mediaFormat);
        }
        catch (IllegalArgumentException e) {
            DebugUtils.logToString(sb, "[" + codec.getName() + "] does not support mime " + mimeType);
        }
        return sb.toString();
    }

    private static void dumpCodecCapabilities(@NonNull StringBuilder sb, @NonNull MediaCodecInfo.CodecCapabilities caps, @NonNull MediaFormat mediaFormat) {
        MediaCodecInfo.EncoderCapabilities encoderCaps;
        MediaCodecInfo.AudioCapabilities audioCaps;
        MediaCodecInfo.VideoCapabilities videoCaps;
        try {
            DebugUtils.logToString(sb, "[CodecCaps] isFormatSupported = " + caps.isFormatSupported(mediaFormat));
        }
        catch (ClassCastException e) {
            DebugUtils.logToString(sb, "[CodecCaps] isFormatSupported=false");
        }
        DebugUtils.logToString(sb, "[CodecCaps] getDefaultFormat = " + caps.getDefaultFormat());
        if (caps.profileLevels != null) {
            StringBuilder stringBuilder = new StringBuilder("[");
            ArrayList<String> profileLevelsStr = new ArrayList<String>();
            for (MediaCodecInfo.CodecProfileLevel profileLevel : caps.profileLevels) {
                profileLevelsStr.add(DebugUtils.toString(profileLevel));
            }
            stringBuilder.append(TextUtils.join((CharSequence)", ", profileLevelsStr)).append("]");
            DebugUtils.logToString(sb, "[CodecCaps] profileLevels = " + stringBuilder);
        }
        if (caps.colorFormats != null) {
            DebugUtils.logToString(sb, "[CodecCaps] colorFormats = " + Arrays.toString(caps.colorFormats));
        }
        if ((videoCaps = caps.getVideoCapabilities()) != null) {
            DebugUtils.dumpVideoCapabilities(sb, videoCaps, mediaFormat);
        }
        if ((audioCaps = caps.getAudioCapabilities()) != null) {
            DebugUtils.dumpAudioCapabilities(sb, audioCaps, mediaFormat);
        }
        if ((encoderCaps = caps.getEncoderCapabilities()) != null) {
            DebugUtils.dumpEncoderCapabilities(sb, encoderCaps, mediaFormat);
        }
    }

    private static void dumpVideoCapabilities(@NonNull StringBuilder sb, @NonNull MediaCodecInfo.VideoCapabilities caps, @NonNull MediaFormat mediaFormat) {
        int frameRate;
        int height;
        int width;
        DebugUtils.logToString(sb, "[VideoCaps] getBitrateRange = " + caps.getBitrateRange());
        DebugUtils.logToString(sb, "[VideoCaps] getSupportedWidths = " + caps.getSupportedWidths() + ", getWidthAlignment = " + caps.getWidthAlignment());
        DebugUtils.logToString(sb, "[VideoCaps] getSupportedHeights = " + caps.getSupportedHeights() + ", getHeightAlignment = " + caps.getHeightAlignment());
        boolean hasSize = true;
        try {
            width = mediaFormat.getInteger("width");
            height = mediaFormat.getInteger("height");
            Preconditions.checkArgument((width > 0 && height > 0 ? 1 : 0) != 0);
        }
        catch (IllegalArgumentException | NullPointerException e) {
            DebugUtils.logToString(sb, "[VideoCaps] mediaFormat does not contain valid width and height");
            height = 0;
            width = 0;
            hasSize = false;
        }
        if (hasSize) {
            try {
                DebugUtils.logToString(sb, "[VideoCaps] getSupportedHeightsFor " + width + " = " + caps.getSupportedHeightsFor(width));
            }
            catch (IllegalArgumentException e) {
                DebugUtils.logToString(sb, "[VideoCaps] could not getSupportedHeightsFor " + width);
            }
            try {
                DebugUtils.logToString(sb, "[VideoCaps] getSupportedWidthsFor " + height + " = " + caps.getSupportedWidthsFor(height));
            }
            catch (IllegalArgumentException e) {
                DebugUtils.logToString(sb, "[VideoCaps] could not getSupportedWidthsFor " + height);
            }
            DebugUtils.logToString(sb, "[VideoCaps] isSizeSupported for " + width + "x" + height + " = " + caps.isSizeSupported(width, height));
        }
        DebugUtils.logToString(sb, "[VideoCaps] getSupportedFrameRates = " + caps.getSupportedFrameRates());
        try {
            frameRate = mediaFormat.getInteger("frame-rate");
            Preconditions.checkArgument((frameRate > 0 ? 1 : 0) != 0);
        }
        catch (IllegalArgumentException | NullPointerException e) {
            DebugUtils.logToString(sb, "[VideoCaps] mediaFormat does not contain frame rate");
            frameRate = 0;
        }
        if (hasSize) {
            DebugUtils.logToString(sb, "[VideoCaps] getSupportedFrameRatesFor " + width + "x" + height + " = " + caps.getSupportedFrameRatesFor(width, height));
        }
        if (hasSize && frameRate > 0) {
            DebugUtils.logToString(sb, "[VideoCaps] areSizeAndRateSupported for " + width + "x" + height + ", " + frameRate + " = " + caps.areSizeAndRateSupported(width, height, (double)frameRate));
        }
    }

    private static void dumpAudioCapabilities(@NonNull StringBuilder sb, @NonNull MediaCodecInfo.AudioCapabilities caps, @NonNull MediaFormat mediaFormat) {
        DebugUtils.logToString(sb, "[AudioCaps] getBitrateRange = " + caps.getBitrateRange());
        DebugUtils.logToString(sb, "[AudioCaps] getMaxInputChannelCount = " + caps.getMaxInputChannelCount());
        if (Build.VERSION.SDK_INT >= 31) {
            DebugUtils.logToString(sb, "[AudioCaps] getMinInputChannelCount = " + Api31Impl.getMinInputChannelCount(caps));
            DebugUtils.logToString(sb, "[AudioCaps] getInputChannelCountRanges = " + Arrays.toString(Api31Impl.getInputChannelCountRanges(caps)));
        }
        DebugUtils.logToString(sb, "[AudioCaps] getSupportedSampleRateRanges = " + Arrays.toString(caps.getSupportedSampleRateRanges()));
        DebugUtils.logToString(sb, "[AudioCaps] getSupportedSampleRates = " + Arrays.toString(caps.getSupportedSampleRates()));
        try {
            int sampleRate = mediaFormat.getInteger("sample-rate");
            DebugUtils.logToString(sb, "[AudioCaps] isSampleRateSupported for " + sampleRate + " = " + caps.isSampleRateSupported(sampleRate));
        }
        catch (IllegalArgumentException | NullPointerException e) {
            DebugUtils.logToString(sb, "[AudioCaps] mediaFormat does not contain sample rate");
        }
    }

    private static void dumpEncoderCapabilities(@NonNull StringBuilder sb, @NonNull MediaCodecInfo.EncoderCapabilities caps, @NonNull MediaFormat mediaFormat) {
        DebugUtils.logToString(sb, "[EncoderCaps] getComplexityRange = " + caps.getComplexityRange());
        if (Build.VERSION.SDK_INT >= 28) {
            DebugUtils.logToString(sb, "[EncoderCaps] getQualityRange = " + Api28Impl.getQualityRange(caps));
        }
        try {
            int bitrateMode = mediaFormat.getInteger("bitrate-mode");
            DebugUtils.logToString(sb, "[EncoderCaps] isBitrateModeSupported = " + caps.isBitrateModeSupported(bitrateMode));
        }
        catch (IllegalArgumentException | NullPointerException e) {
            DebugUtils.logToString(sb, "[EncoderCaps] mediaFormat does not contain bitrate mode");
        }
    }

    private static void logToString(@NonNull StringBuilder sb, @NonNull String message) {
        sb.append(message);
        sb.append("\n");
    }

    private static void stringToLog(@NonNull String log) {
        if (Logger.isInfoEnabled((String)TAG)) {
            Scanner scan = new Scanner(log);
            while (scan.hasNextLine()) {
                Logger.i((String)TAG, (String)scan.nextLine());
            }
        }
    }

    @NonNull
    private static String toString(@Nullable MediaCodecInfo.CodecProfileLevel codecProfileLevel) {
        if (codecProfileLevel == null) {
            return "null";
        }
        return String.format("{level=%d, profile=%d}", codecProfileLevel.level, codecProfileLevel.profile);
    }
}

