/*
 * Decompiled with CFR 0.152.
 */
package androidx.camera.video.internal.encoder;

import android.annotation.SuppressLint;
import android.media.MediaCodec;
import android.media.MediaCodecList;
import android.media.MediaFormat;
import android.os.Bundle;
import android.util.Range;
import android.view.Surface;
import androidx.annotation.DoNotInline;
import androidx.annotation.GuardedBy;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;
import androidx.camera.core.Logger;
import androidx.camera.core.impl.Observable;
import androidx.camera.core.impl.utils.executor.CameraXExecutors;
import androidx.camera.core.impl.utils.futures.FutureCallback;
import androidx.camera.core.impl.utils.futures.Futures;
import androidx.camera.video.internal.BufferProvider;
import androidx.camera.video.internal.DebugUtils;
import androidx.camera.video.internal.compat.quirk.AudioEncoderIgnoresInputTimestampQuirk;
import androidx.camera.video.internal.compat.quirk.CameraUseInconsistentTimebaseQuirk;
import androidx.camera.video.internal.compat.quirk.DeviceQuirks;
import androidx.camera.video.internal.compat.quirk.EncoderNotUsePersistentInputSurfaceQuirk;
import androidx.camera.video.internal.compat.quirk.VideoEncoderSuspendDoesNotIncludeSuspendTimeQuirk;
import androidx.camera.video.internal.encoder.AudioEncoderConfig;
import androidx.camera.video.internal.encoder.EncodeException;
import androidx.camera.video.internal.encoder.EncodedDataImpl;
import androidx.camera.video.internal.encoder.Encoder;
import androidx.camera.video.internal.encoder.EncoderCallback;
import androidx.camera.video.internal.encoder.EncoderConfig;
import androidx.camera.video.internal.encoder.InputBuffer;
import androidx.camera.video.internal.encoder.InputBufferImpl;
import androidx.camera.video.internal.encoder.InvalidConfigException;
import androidx.camera.video.internal.encoder.VideoEncoderConfig;
import androidx.camera.video.internal.workaround.CorrectVideoTimeByTimebase;
import androidx.camera.video.internal.workaround.EncoderFinder;
import androidx.concurrent.futures.CallbackToFutureAdapter;
import androidx.core.util.Preconditions;
import com.google.common.util.concurrent.ListenableFuture;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Deque;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.Executor;
import java.util.concurrent.Future;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;

@RequiresApi(value=21)
public class EncoderImpl
implements Encoder {
    private static final boolean DEBUG = false;
    private static final long NO_LIMIT_LONG = Long.MAX_VALUE;
    private static final Range<Long> NO_RANGE = Range.create((Comparable)Long.valueOf(Long.MAX_VALUE), (Comparable)Long.valueOf(Long.MAX_VALUE));
    private static final long STOP_TIMEOUT_MS = 1000L;
    private static final long TIMESTAMP_ANY = -1L;
    final String mTag;
    final Object mLock = new Object();
    final boolean mIsVideoEncoder;
    private final MediaFormat mMediaFormat;
    final MediaCodec mMediaCodec;
    final Encoder.EncoderInput mEncoderInput;
    final Executor mEncoderExecutor;
    final Queue<Integer> mFreeInputBufferIndexQueue = new ArrayDeque<Integer>();
    private final Queue<CallbackToFutureAdapter.Completer<InputBuffer>> mAcquisitionQueue = new ArrayDeque<CallbackToFutureAdapter.Completer<InputBuffer>>();
    private final Set<InputBuffer> mInputBufferSet = new HashSet<InputBuffer>();
    final Set<EncodedDataImpl> mEncodedDataSet = new HashSet<EncodedDataImpl>();
    final Deque<Range<Long>> mActivePauseResumeTimeRanges = new ArrayDeque<Range<Long>>();
    @GuardedBy(value="mLock")
    EncoderCallback mEncoderCallback = EncoderCallback.EMPTY;
    @GuardedBy(value="mLock")
    Executor mEncoderCallbackExecutor = CameraXExecutors.directExecutor();
    InternalState mState;
    Range<Long> mStartStopTimeRangeUs = NO_RANGE;
    long mTotalPausedDurationUs = 0L;
    boolean mPendingCodecStop = false;
    Long mLastDataStopTimestamp = null;
    Future<?> mStopTimeoutFuture = null;
    private boolean mIsFlushedAfterEndOfStream = false;
    private boolean mSourceStoppedSignalled = false;
    final EncoderFinder mEncoderFinder = new EncoderFinder();

    public EncoderImpl(@NonNull Executor executor, @NonNull EncoderConfig encoderConfig) throws InvalidConfigException {
        Preconditions.checkNotNull((Object)executor);
        Preconditions.checkNotNull((Object)encoderConfig);
        this.mEncoderExecutor = CameraXExecutors.newSequentialExecutor((Executor)executor);
        if (encoderConfig instanceof AudioEncoderConfig) {
            this.mTag = "AudioEncoder";
            this.mIsVideoEncoder = false;
            this.mEncoderInput = new ByteBufferInput();
        } else if (encoderConfig instanceof VideoEncoderConfig) {
            this.mTag = "VideoEncoder";
            this.mIsVideoEncoder = true;
            this.mEncoderInput = new SurfaceInput();
        } else {
            throw new InvalidConfigException("Unknown encoder config type");
        }
        this.mMediaFormat = encoderConfig.toMediaFormat();
        Logger.d((String)this.mTag, (String)("mMediaFormat = " + this.mMediaFormat));
        this.mMediaCodec = this.mEncoderFinder.findEncoder(this.mMediaFormat, new MediaCodecList(1));
        Logger.i((String)this.mTag, (String)("Selected encoder: " + this.mMediaCodec.getName()));
        try {
            this.reset();
        }
        catch (MediaCodec.CodecException e) {
            throw new InvalidConfigException(e);
        }
        this.setState(InternalState.CONFIGURED);
    }

    private void reset() {
        this.mStartStopTimeRangeUs = NO_RANGE;
        this.mTotalPausedDurationUs = 0L;
        this.mActivePauseResumeTimeRanges.clear();
        this.mFreeInputBufferIndexQueue.clear();
        for (CallbackToFutureAdapter.Completer completer : this.mAcquisitionQueue) {
            completer.setCancelled();
        }
        this.mAcquisitionQueue.clear();
        this.mMediaCodec.reset();
        this.mIsFlushedAfterEndOfStream = false;
        this.mSourceStoppedSignalled = false;
        this.mPendingCodecStop = false;
        if (this.mStopTimeoutFuture != null) {
            this.mStopTimeoutFuture.cancel(true);
            this.mStopTimeoutFuture = null;
        }
        this.mMediaCodec.setCallback((MediaCodec.Callback)new MediaCodecCallback());
        this.mMediaCodec.configure(this.mMediaFormat, null, null, 1);
        if (this.mEncoderInput instanceof SurfaceInput) {
            ((SurfaceInput)this.mEncoderInput).resetSurface();
        }
    }

    @Override
    @NonNull
    public Encoder.EncoderInput getInput() {
        return this.mEncoderInput;
    }

    @Override
    public void start() {
        long startTriggerTimeUs = EncoderImpl.generatePresentationTimeUs();
        this.mEncoderExecutor.execute(() -> {
            switch (this.mState) {
                case CONFIGURED: {
                    this.mLastDataStopTimestamp = null;
                    long startTimeUs = startTriggerTimeUs;
                    Logger.d((String)this.mTag, (String)("Start on " + DebugUtils.readableUs(startTimeUs)));
                    try {
                        if (this.mIsFlushedAfterEndOfStream) {
                            this.reset();
                        }
                        this.mStartStopTimeRangeUs = Range.create((Comparable)Long.valueOf(startTimeUs), (Comparable)Long.valueOf(Long.MAX_VALUE));
                        this.mMediaCodec.start();
                    }
                    catch (MediaCodec.CodecException e) {
                        this.handleEncodeError(e);
                        return;
                    }
                    if (this.mEncoderInput instanceof ByteBufferInput) {
                        ((ByteBufferInput)this.mEncoderInput).setActive(true);
                    }
                    this.setState(InternalState.STARTED);
                    break;
                }
                case PAUSED: {
                    this.mLastDataStopTimestamp = null;
                    Range<Long> pauseRange = this.mActivePauseResumeTimeRanges.removeLast();
                    Preconditions.checkState((pauseRange != null && (Long)pauseRange.getUpper() == Long.MAX_VALUE ? 1 : 0) != 0, (String)"There should be a \"pause\" before \"resume\"");
                    long pauseTimeUs = (Long)pauseRange.getLower();
                    long resumeTimeUs = startTriggerTimeUs;
                    this.mActivePauseResumeTimeRanges.addLast((Range<Long>)Range.create((Comparable)Long.valueOf(pauseTimeUs), (Comparable)Long.valueOf(resumeTimeUs)));
                    Logger.d((String)this.mTag, (String)("Resume on " + DebugUtils.readableUs(resumeTimeUs) + "\nPaused duration = " + DebugUtils.readableUs(resumeTimeUs - pauseTimeUs)));
                    if (!(!this.mIsVideoEncoder && DeviceQuirks.get(AudioEncoderIgnoresInputTimestampQuirk.class) != null || this.mIsVideoEncoder && DeviceQuirks.get(VideoEncoderSuspendDoesNotIncludeSuspendTimeQuirk.class) != null)) {
                        this.setMediaCodecPaused(false);
                        if (this.mEncoderInput instanceof ByteBufferInput) {
                            ((ByteBufferInput)this.mEncoderInput).setActive(true);
                        }
                    }
                    if (this.mIsVideoEncoder) {
                        this.requestKeyFrameToMediaCodec();
                    }
                    this.setState(InternalState.STARTED);
                    break;
                }
                case STARTED: 
                case PENDING_START: 
                case ERROR: {
                    break;
                }
                case STOPPING: 
                case PENDING_START_PAUSED: {
                    this.setState(InternalState.PENDING_START);
                    break;
                }
                case PENDING_RELEASE: 
                case RELEASED: {
                    throw new IllegalStateException("Encoder is released");
                }
                default: {
                    throw new IllegalStateException("Unknown state: " + (Object)((Object)this.mState));
                }
            }
        });
    }

    @Override
    public void stop() {
        this.stop(-1L);
    }

    @Override
    public void stop(long expectedStopTimeUs) {
        long stopTriggerTimeUs = EncoderImpl.generatePresentationTimeUs();
        this.mEncoderExecutor.execute(() -> {
            switch (this.mState) {
                case CONFIGURED: 
                case STOPPING: 
                case ERROR: {
                    break;
                }
                case STARTED: 
                case PAUSED: {
                    long stopTimeUs;
                    InternalState currentState = this.mState;
                    this.setState(InternalState.STOPPING);
                    long startTimeUs = (Long)this.mStartStopTimeRangeUs.getLower();
                    if (startTimeUs == Long.MAX_VALUE) {
                        throw new AssertionError((Object)"There should be a \"start\" before \"stop\"");
                    }
                    if (expectedStopTimeUs == -1L) {
                        stopTimeUs = stopTriggerTimeUs;
                    } else if (expectedStopTimeUs < startTimeUs) {
                        Logger.w((String)this.mTag, (String)"The expected stop time is less than the start time. Use current time as stop time.");
                        stopTimeUs = stopTriggerTimeUs;
                    } else {
                        stopTimeUs = expectedStopTimeUs;
                    }
                    if (stopTimeUs < startTimeUs) {
                        throw new AssertionError((Object)"The start time should be before the stop time.");
                    }
                    this.mStartStopTimeRangeUs = Range.create((Comparable)Long.valueOf(startTimeUs), (Comparable)Long.valueOf(stopTimeUs));
                    Logger.d((String)this.mTag, (String)("Stop on " + DebugUtils.readableUs(stopTimeUs)));
                    if (currentState == InternalState.PAUSED && this.mLastDataStopTimestamp != null) {
                        this.signalCodecStop();
                        break;
                    }
                    this.mPendingCodecStop = true;
                    this.mStopTimeoutFuture = CameraXExecutors.mainThreadExecutor().schedule(() -> this.mEncoderExecutor.execute(() -> {
                        if (this.mPendingCodecStop) {
                            Logger.w((String)this.mTag, (String)"The data didn't reach the expected timestamp before timeout, stop the codec.");
                            this.mLastDataStopTimestamp = null;
                            this.signalCodecStop();
                            this.mPendingCodecStop = false;
                        }
                    }), 1000L, TimeUnit.MILLISECONDS);
                    break;
                }
                case PENDING_START_PAUSED: 
                case PENDING_START: {
                    this.setState(InternalState.CONFIGURED);
                    break;
                }
                case PENDING_RELEASE: 
                case RELEASED: {
                    throw new IllegalStateException("Encoder is released");
                }
                default: {
                    throw new IllegalStateException("Unknown state: " + (Object)((Object)this.mState));
                }
            }
        });
    }

    void signalCodecStop() {
        if (this.mEncoderInput instanceof ByteBufferInput) {
            ((ByteBufferInput)this.mEncoderInput).setActive(false);
            ArrayList<ListenableFuture<Void>> futures = new ArrayList<ListenableFuture<Void>>();
            for (InputBuffer inputBuffer : this.mInputBufferSet) {
                futures.add(inputBuffer.getTerminationFuture());
            }
            Futures.successfulAsList(futures).addListener(this::signalEndOfInputStream, this.mEncoderExecutor);
        } else if (this.mEncoderInput instanceof SurfaceInput) {
            try {
                this.mMediaCodec.signalEndOfInputStream();
            }
            catch (MediaCodec.CodecException e) {
                this.handleEncodeError(e);
            }
        }
    }

    @Override
    public void pause() {
        long pauseTriggerTimeUs = EncoderImpl.generatePresentationTimeUs();
        this.mEncoderExecutor.execute(() -> {
            switch (this.mState) {
                case CONFIGURED: 
                case PAUSED: 
                case STOPPING: 
                case PENDING_START_PAUSED: 
                case ERROR: {
                    break;
                }
                case PENDING_START: {
                    this.setState(InternalState.PENDING_START_PAUSED);
                    break;
                }
                case STARTED: {
                    long pauseTimeUs = pauseTriggerTimeUs;
                    Logger.d((String)this.mTag, (String)("Pause on " + DebugUtils.readableUs(pauseTimeUs)));
                    this.mActivePauseResumeTimeRanges.addLast((Range<Long>)Range.create((Comparable)Long.valueOf(pauseTimeUs), (Comparable)Long.valueOf(Long.MAX_VALUE)));
                    this.setState(InternalState.PAUSED);
                    break;
                }
                case PENDING_RELEASE: 
                case RELEASED: {
                    throw new IllegalStateException("Encoder is released");
                }
                default: {
                    throw new IllegalStateException("Unknown state: " + (Object)((Object)this.mState));
                }
            }
        });
    }

    @Override
    public void release() {
        this.mEncoderExecutor.execute(() -> {
            switch (this.mState) {
                case CONFIGURED: 
                case STARTED: 
                case PAUSED: 
                case ERROR: {
                    this.releaseInternal();
                    break;
                }
                case STOPPING: 
                case PENDING_START_PAUSED: 
                case PENDING_START: {
                    this.setState(InternalState.PENDING_RELEASE);
                    break;
                }
                case PENDING_RELEASE: 
                case RELEASED: {
                    break;
                }
                default: {
                    throw new IllegalStateException("Unknown state: " + (Object)((Object)this.mState));
                }
            }
        });
    }

    public void signalSourceStopped() {
        this.mEncoderExecutor.execute(() -> {
            this.mSourceStoppedSignalled = true;
            if (this.mIsFlushedAfterEndOfStream) {
                this.mMediaCodec.stop();
                this.reset();
            }
        });
    }

    private void releaseInternal() {
        if (this.mIsFlushedAfterEndOfStream) {
            this.mMediaCodec.stop();
            this.mIsFlushedAfterEndOfStream = false;
        }
        this.mMediaCodec.release();
        if (this.mEncoderInput instanceof SurfaceInput) {
            ((SurfaceInput)this.mEncoderInput).releaseSurface();
        }
        this.setState(InternalState.RELEASED);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void setEncoderCallback(@NonNull EncoderCallback encoderCallback, @NonNull Executor executor) {
        Object object = this.mLock;
        synchronized (object) {
            this.mEncoderCallback = encoderCallback;
            this.mEncoderCallbackExecutor = executor;
        }
    }

    @Override
    public void requestKeyFrame() {
        this.mEncoderExecutor.execute(() -> {
            switch (this.mState) {
                case STARTED: {
                    this.requestKeyFrameToMediaCodec();
                    break;
                }
                case CONFIGURED: 
                case PAUSED: 
                case STOPPING: 
                case PENDING_START_PAUSED: 
                case PENDING_START: 
                case ERROR: {
                    break;
                }
                case PENDING_RELEASE: 
                case RELEASED: {
                    throw new IllegalStateException("Encoder is released");
                }
            }
        });
    }

    private void setState(InternalState state) {
        if (this.mState == state) {
            return;
        }
        Logger.d((String)this.mTag, (String)("Transitioning encoder internal state: " + (Object)((Object)this.mState) + " --> " + (Object)((Object)state)));
        this.mState = state;
    }

    void setMediaCodecPaused(boolean paused) {
        Bundle bundle = new Bundle();
        bundle.putInt("drop-input-frames", paused ? 1 : 0);
        this.mMediaCodec.setParameters(bundle);
    }

    void requestKeyFrameToMediaCodec() {
        Bundle bundle = new Bundle();
        bundle.putInt("request-sync", 0);
        this.mMediaCodec.setParameters(bundle);
    }

    private void signalEndOfInputStream() {
        Futures.addCallback(this.acquireInputBuffer(), (FutureCallback)new FutureCallback<InputBuffer>(){

            public void onSuccess(InputBuffer inputBuffer) {
                inputBuffer.setPresentationTimeUs(EncoderImpl.generatePresentationTimeUs());
                inputBuffer.setEndOfStream(true);
                inputBuffer.submit();
                Futures.addCallback(inputBuffer.getTerminationFuture(), (FutureCallback)new FutureCallback<Void>(){

                    public void onSuccess(@Nullable Void result) {
                    }

                    public void onFailure(Throwable t) {
                        if (t instanceof MediaCodec.CodecException) {
                            EncoderImpl.this.handleEncodeError((MediaCodec.CodecException)t);
                        } else {
                            EncoderImpl.this.handleEncodeError(0, t.getMessage(), t);
                        }
                    }
                }, (Executor)EncoderImpl.this.mEncoderExecutor);
            }

            public void onFailure(Throwable t) {
                EncoderImpl.this.handleEncodeError(0, "Unable to acquire InputBuffer.", t);
            }
        }, (Executor)this.mEncoderExecutor);
    }

    void handleEncodeError(@NonNull MediaCodec.CodecException e) {
        this.handleEncodeError(1, e.getMessage(), e);
    }

    void handleEncodeError(int error, @Nullable String message, @Nullable Throwable throwable) {
        switch (this.mState) {
            case CONFIGURED: {
                this.notifyError(error, message, throwable);
                this.reset();
                break;
            }
            case STARTED: 
            case PAUSED: 
            case STOPPING: 
            case PENDING_START_PAUSED: 
            case PENDING_START: 
            case PENDING_RELEASE: {
                this.setState(InternalState.ERROR);
                this.stopMediaCodec(() -> this.notifyError(error, message, throwable));
                break;
            }
            case ERROR: {
                Logger.w((String)this.mTag, (String)("Get more than one error: " + message + "(" + error + ")"), (Throwable)throwable);
                break;
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void notifyError(int error, @Nullable String message, @Nullable Throwable throwable) {
        Executor executor;
        EncoderCallback callback;
        Object object = this.mLock;
        synchronized (object) {
            callback = this.mEncoderCallback;
            executor = this.mEncoderCallbackExecutor;
        }
        try {
            executor.execute(() -> callback.onEncodeError(new EncodeException(error, message, throwable)));
        }
        catch (RejectedExecutionException e) {
            Logger.e((String)this.mTag, (String)"Unable to post to the supplied executor.", (Throwable)e);
        }
    }

    void stopMediaCodec(@Nullable Runnable afterStop) {
        ArrayList<ListenableFuture<Void>> futures = new ArrayList<ListenableFuture<Void>>();
        for (EncodedDataImpl dataToClose : this.mEncodedDataSet) {
            futures.add(dataToClose.getClosedFuture());
        }
        for (InputBuffer inputBuffer : this.mInputBufferSet) {
            futures.add(inputBuffer.getTerminationFuture());
        }
        Futures.successfulAsList(futures).addListener(() -> {
            if (this.mEncoderInput instanceof SurfaceInput && !this.mSourceStoppedSignalled) {
                this.mMediaCodec.flush();
                this.mIsFlushedAfterEndOfStream = true;
            } else {
                this.mMediaCodec.stop();
            }
            if (afterStop != null) {
                afterStop.run();
            }
            this.handleStopped();
        }, this.mEncoderExecutor);
    }

    void handleStopped() {
        if (this.mState == InternalState.PENDING_RELEASE) {
            this.releaseInternal();
        } else {
            InternalState oldState = this.mState;
            if (!this.mIsFlushedAfterEndOfStream) {
                this.reset();
            }
            this.setState(InternalState.CONFIGURED);
            if (oldState == InternalState.PENDING_START || oldState == InternalState.PENDING_START_PAUSED) {
                this.start();
                if (oldState == InternalState.PENDING_START_PAUSED) {
                    this.pause();
                }
            }
        }
    }

    void updateTotalPausedDuration(long bufferPresentationTimeUs) {
        Range<Long> pauseRange;
        while (!this.mActivePauseResumeTimeRanges.isEmpty() && bufferPresentationTimeUs > (Long)(pauseRange = this.mActivePauseResumeTimeRanges.getFirst()).getUpper()) {
            this.mActivePauseResumeTimeRanges.removeFirst();
            this.mTotalPausedDurationUs += (Long)pauseRange.getUpper() - (Long)pauseRange.getLower();
            Logger.d((String)this.mTag, (String)("Total paused duration = " + DebugUtils.readableUs(this.mTotalPausedDurationUs)));
        }
    }

    long getAdjustedTimeUs(@NonNull MediaCodec.BufferInfo bufferInfo) {
        long adjustedTimeUs = this.mTotalPausedDurationUs > 0L ? bufferInfo.presentationTimeUs - this.mTotalPausedDurationUs : bufferInfo.presentationTimeUs;
        return adjustedTimeUs;
    }

    boolean isInPauseRange(long timeUs) {
        for (Range<Long> range : this.mActivePauseResumeTimeRanges) {
            if (range.contains((Comparable)Long.valueOf(timeUs))) {
                return true;
            }
            if (timeUs >= (Long)range.getLower()) continue;
            return false;
        }
        return false;
    }

    @NonNull
    ListenableFuture<InputBuffer> acquireInputBuffer() {
        switch (this.mState) {
            case CONFIGURED: {
                return Futures.immediateFailedFuture((Throwable)new IllegalStateException("Encoder is not started yet."));
            }
            case STARTED: 
            case PAUSED: 
            case STOPPING: 
            case PENDING_START_PAUSED: 
            case PENDING_START: 
            case PENDING_RELEASE: {
                AtomicReference ref = new AtomicReference();
                ListenableFuture future = CallbackToFutureAdapter.getFuture(completer -> {
                    ref.set(completer);
                    return "acquireInputBuffer";
                });
                CallbackToFutureAdapter.Completer completer2 = (CallbackToFutureAdapter.Completer)Preconditions.checkNotNull((Object)((CallbackToFutureAdapter.Completer)ref.get()));
                this.mAcquisitionQueue.offer((CallbackToFutureAdapter.Completer<InputBuffer>)completer2);
                completer2.addCancellationListener(() -> this.mAcquisitionQueue.remove(completer2), this.mEncoderExecutor);
                this.matchAcquisitionsAndFreeBufferIndexes();
                return future;
            }
            case ERROR: {
                return Futures.immediateFailedFuture((Throwable)new IllegalStateException("Encoder is in error state."));
            }
            case RELEASED: {
                return Futures.immediateFailedFuture((Throwable)new IllegalStateException("Encoder is released."));
            }
        }
        throw new IllegalStateException("Unknown state: " + (Object)((Object)this.mState));
    }

    void matchAcquisitionsAndFreeBufferIndexes() {
        while (!this.mAcquisitionQueue.isEmpty() && !this.mFreeInputBufferIndexQueue.isEmpty()) {
            InputBufferImpl inputBuffer;
            CallbackToFutureAdapter.Completer<InputBuffer> completer = this.mAcquisitionQueue.poll();
            int bufferIndex = this.mFreeInputBufferIndexQueue.poll();
            try {
                inputBuffer = new InputBufferImpl(this.mMediaCodec, bufferIndex);
            }
            catch (MediaCodec.CodecException e) {
                this.handleEncodeError(e);
                return;
            }
            if (completer.set((Object)inputBuffer)) {
                this.mInputBufferSet.add(inputBuffer);
                inputBuffer.getTerminationFuture().addListener(() -> this.mInputBufferSet.remove(inputBuffer), this.mEncoderExecutor);
                continue;
            }
            inputBuffer.cancel();
        }
    }

    static long generatePresentationTimeUs() {
        return TimeUnit.NANOSECONDS.toMicros(System.nanoTime());
    }

    static boolean isKeyFrame(@NonNull MediaCodec.BufferInfo bufferInfo) {
        return (bufferInfo.flags & 1) != 0;
    }

    static boolean isEndOfStream(@NonNull MediaCodec.BufferInfo bufferInfo) {
        return (bufferInfo.flags & 4) != 0;
    }

    @RequiresApi(value=23)
    private static class Api23Impl {
        private Api23Impl() {
        }

        @DoNotInline
        @NonNull
        static Surface createPersistentInputSurface() {
            return MediaCodec.createPersistentInputSurface();
        }

        @DoNotInline
        static void setInputSurface(@NonNull MediaCodec mediaCodec, @NonNull Surface surface) {
            mediaCodec.setInputSurface(surface);
        }
    }

    class ByteBufferInput
    implements Encoder.ByteBufferInput {
        private final Map<Observable.Observer<? super BufferProvider.State>, Executor> mStateObservers = new LinkedHashMap<Observable.Observer<? super BufferProvider.State>, Executor>();
        private BufferProvider.State mBufferProviderState = BufferProvider.State.INACTIVE;
        private final List<ListenableFuture<InputBuffer>> mAcquisitionList = new ArrayList<ListenableFuture<InputBuffer>>();

        ByteBufferInput() {
        }

        @NonNull
        public ListenableFuture<BufferProvider.State> fetchData() {
            return CallbackToFutureAdapter.getFuture(completer -> {
                EncoderImpl.this.mEncoderExecutor.execute(() -> completer.set((Object)this.mBufferProviderState));
                return "fetchData";
            });
        }

        @Override
        @NonNull
        public ListenableFuture<InputBuffer> acquireBuffer() {
            return CallbackToFutureAdapter.getFuture(completer -> {
                EncoderImpl.this.mEncoderExecutor.execute(() -> {
                    if (this.mBufferProviderState == BufferProvider.State.ACTIVE) {
                        ListenableFuture<InputBuffer> future = EncoderImpl.this.acquireInputBuffer();
                        Futures.propagate(future, (CallbackToFutureAdapter.Completer)completer);
                        completer.addCancellationListener(() -> future.cancel(true), CameraXExecutors.directExecutor());
                        this.mAcquisitionList.add(future);
                        future.addListener(() -> this.mAcquisitionList.remove(future), EncoderImpl.this.mEncoderExecutor);
                    } else if (this.mBufferProviderState == BufferProvider.State.INACTIVE) {
                        completer.setException((Throwable)new IllegalStateException("BufferProvider is not active."));
                    } else {
                        completer.setException((Throwable)new IllegalStateException("Unknown state: " + (Object)((Object)this.mBufferProviderState)));
                    }
                });
                return "acquireBuffer";
            });
        }

        public void addObserver(@NonNull Executor executor, @NonNull Observable.Observer<? super BufferProvider.State> observer) {
            EncoderImpl.this.mEncoderExecutor.execute(() -> {
                this.mStateObservers.put((Observable.Observer<? super BufferProvider.State>)((Observable.Observer)Preconditions.checkNotNull((Object)observer)), (Executor)Preconditions.checkNotNull((Object)executor));
                BufferProvider.State state = this.mBufferProviderState;
                executor.execute(() -> observer.onNewData((Object)state));
            });
        }

        public void removeObserver(@NonNull Observable.Observer<? super BufferProvider.State> observer) {
            EncoderImpl.this.mEncoderExecutor.execute(() -> this.mStateObservers.remove(Preconditions.checkNotNull((Object)observer)));
        }

        void setActive(boolean isActive) {
            BufferProvider.State newState;
            BufferProvider.State state = newState = isActive ? BufferProvider.State.ACTIVE : BufferProvider.State.INACTIVE;
            if (this.mBufferProviderState == newState) {
                return;
            }
            this.mBufferProviderState = newState;
            if (newState == BufferProvider.State.INACTIVE) {
                for (ListenableFuture listenableFuture : this.mAcquisitionList) {
                    listenableFuture.cancel(true);
                }
                this.mAcquisitionList.clear();
            }
            for (Map.Entry entry : this.mStateObservers.entrySet()) {
                try {
                    ((Executor)entry.getValue()).execute(() -> ((Observable.Observer)entry.getKey()).onNewData((Object)newState));
                }
                catch (RejectedExecutionException e) {
                    Logger.e((String)EncoderImpl.this.mTag, (String)"Unable to post to the supplied executor.", (Throwable)e);
                }
            }
        }
    }

    @RequiresApi(value=21)
    class SurfaceInput
    implements Encoder.SurfaceInput {
        private final Object mLock = new Object();
        @GuardedBy(value="mLock")
        private Surface mSurface;
        @GuardedBy(value="mLock")
        private final Set<Surface> mObsoleteSurfaces = new HashSet<Surface>();
        @GuardedBy(value="mLock")
        private Encoder.SurfaceInput.OnSurfaceUpdateListener mSurfaceUpdateListener;
        @GuardedBy(value="mLock")
        private Executor mSurfaceUpdateExecutor;

        SurfaceInput() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void setOnSurfaceUpdateListener(@NonNull Executor executor, @NonNull Encoder.SurfaceInput.OnSurfaceUpdateListener listener) {
            Surface surface;
            Object object = this.mLock;
            synchronized (object) {
                this.mSurfaceUpdateListener = (Encoder.SurfaceInput.OnSurfaceUpdateListener)Preconditions.checkNotNull((Object)listener);
                this.mSurfaceUpdateExecutor = (Executor)Preconditions.checkNotNull((Object)executor);
                surface = this.mSurface;
            }
            if (surface != null) {
                this.notifySurfaceUpdate(executor, listener, surface);
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @SuppressLint(value={"NewApi"})
        void resetSurface() {
            Executor executor;
            Encoder.SurfaceInput.OnSurfaceUpdateListener listener;
            Surface surface;
            EncoderNotUsePersistentInputSurfaceQuirk quirk = DeviceQuirks.get(EncoderNotUsePersistentInputSurfaceQuirk.class);
            Object object = this.mLock;
            synchronized (object) {
                if (quirk == null) {
                    surface = this.mSurface == null ? (this.mSurface = Api23Impl.createPersistentInputSurface()) : null;
                    Api23Impl.setInputSurface(EncoderImpl.this.mMediaCodec, this.mSurface);
                } else {
                    if (this.mSurface != null) {
                        this.mObsoleteSurfaces.add(this.mSurface);
                    }
                    surface = this.mSurface = EncoderImpl.this.mMediaCodec.createInputSurface();
                }
                listener = this.mSurfaceUpdateListener;
                executor = this.mSurfaceUpdateExecutor;
            }
            if (surface != null && listener != null && executor != null) {
                this.notifySurfaceUpdate(executor, listener, surface);
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void releaseSurface() {
            HashSet<Surface> obsoleteSurfaces;
            Surface surface;
            Iterator iterator = this.mLock;
            synchronized (iterator) {
                surface = this.mSurface;
                this.mSurface = null;
                obsoleteSurfaces = new HashSet<Surface>(this.mObsoleteSurfaces);
                this.mObsoleteSurfaces.clear();
            }
            if (surface != null) {
                surface.release();
            }
            for (Surface obsoleteSurface : obsoleteSurfaces) {
                obsoleteSurface.release();
            }
        }

        private void notifySurfaceUpdate(@NonNull Executor executor, @NonNull Encoder.SurfaceInput.OnSurfaceUpdateListener listener, @NonNull Surface surface) {
            try {
                executor.execute(() -> listener.onSurfaceUpdate(surface));
            }
            catch (RejectedExecutionException e) {
                Logger.e((String)EncoderImpl.this.mTag, (String)"Unable to post to the supplied executor.", (Throwable)e);
            }
        }
    }

    @RequiresApi(value=21)
    class MediaCodecCallback
    extends MediaCodec.Callback {
        @Nullable
        private final CorrectVideoTimeByTimebase mCorrectVideoTimestamp;
        private boolean mHasSendStartCallback = false;
        private boolean mHasFirstData = false;
        private boolean mHasEndData = false;
        private long mLastPresentationTimeUs = 0L;
        private long mLastSentPresentationTimeUs = 0L;
        private boolean mIsOutputBufferInPauseState = false;
        private boolean mIsKeyFrameRequired = false;

        MediaCodecCallback() {
            this.mCorrectVideoTimestamp = EncoderImpl.this.mIsVideoEncoder && DeviceQuirks.get(CameraUseInconsistentTimebaseQuirk.class) != null ? new CorrectVideoTimeByTimebase() : null;
        }

        public void onInputBufferAvailable(MediaCodec mediaCodec, int index) {
            EncoderImpl.this.mEncoderExecutor.execute(() -> {
                switch (EncoderImpl.this.mState) {
                    case STARTED: 
                    case PAUSED: 
                    case STOPPING: 
                    case PENDING_START_PAUSED: 
                    case PENDING_START: 
                    case PENDING_RELEASE: {
                        EncoderImpl.this.mFreeInputBufferIndexQueue.offer(index);
                        EncoderImpl.this.matchAcquisitionsAndFreeBufferIndexes();
                        break;
                    }
                    case CONFIGURED: 
                    case ERROR: 
                    case RELEASED: {
                        break;
                    }
                    default: {
                        throw new IllegalStateException("Unknown state: " + (Object)((Object)EncoderImpl.this.mState));
                    }
                }
            });
        }

        public void onOutputBufferAvailable(@NonNull MediaCodec mediaCodec, int index, @NonNull MediaCodec.BufferInfo bufferInfo) {
            EncoderImpl.this.mEncoderExecutor.execute(() -> {
                switch (EncoderImpl.this.mState) {
                    case STARTED: 
                    case PAUSED: 
                    case STOPPING: 
                    case PENDING_START_PAUSED: 
                    case PENDING_START: 
                    case PENDING_RELEASE: {
                        Executor executor;
                        EncoderCallback encoderCallback;
                        Object object = EncoderImpl.this.mLock;
                        synchronized (object) {
                            encoderCallback = EncoderImpl.this.mEncoderCallback;
                            executor = EncoderImpl.this.mEncoderCallbackExecutor;
                        }
                        if (this.mCorrectVideoTimestamp != null) {
                            this.mCorrectVideoTimestamp.correctTimestamp(bufferInfo);
                        }
                        if (!this.mHasSendStartCallback) {
                            this.mHasSendStartCallback = true;
                            try {
                                executor.execute(encoderCallback::onEncodeStart);
                            }
                            catch (RejectedExecutionException e) {
                                Logger.e((String)EncoderImpl.this.mTag, (String)"Unable to post to the supplied executor.", (Throwable)e);
                            }
                        }
                        if (!this.checkBufferInfo(bufferInfo)) {
                            long adjustedTimeUs;
                            if (!this.mHasFirstData) {
                                this.mHasFirstData = true;
                            }
                            if (bufferInfo.presentationTimeUs != (adjustedTimeUs = EncoderImpl.this.getAdjustedTimeUs(bufferInfo))) {
                                Preconditions.checkState((adjustedTimeUs > this.mLastSentPresentationTimeUs ? 1 : 0) != 0);
                                bufferInfo.presentationTimeUs = adjustedTimeUs;
                            }
                            this.mLastSentPresentationTimeUs = bufferInfo.presentationTimeUs;
                            try {
                                EncodedDataImpl encodedData = new EncodedDataImpl(mediaCodec, index, bufferInfo);
                                this.sendEncodedData(encodedData, encoderCallback, executor);
                            }
                            catch (MediaCodec.CodecException e) {
                                EncoderImpl.this.handleEncodeError(e);
                                return;
                            }
                        }
                        try {
                            EncoderImpl.this.mMediaCodec.releaseOutputBuffer(index, false);
                        }
                        catch (MediaCodec.CodecException e) {
                            EncoderImpl.this.handleEncodeError(e);
                            return;
                        }
                        if (this.mHasEndData || !EncoderImpl.isEndOfStream(bufferInfo)) break;
                        this.mHasEndData = true;
                        EncoderImpl.this.stopMediaCodec(() -> {
                            if (EncoderImpl.this.mState == InternalState.ERROR) {
                                return;
                            }
                            try {
                                executor.execute(encoderCallback::onEncodeStop);
                            }
                            catch (RejectedExecutionException e) {
                                Logger.e((String)EncoderImpl.this.mTag, (String)"Unable to post to the supplied executor.", (Throwable)e);
                            }
                        });
                        break;
                    }
                    case CONFIGURED: 
                    case ERROR: 
                    case RELEASED: {
                        break;
                    }
                    default: {
                        throw new IllegalStateException("Unknown state: " + (Object)((Object)EncoderImpl.this.mState));
                    }
                }
            });
        }

        private void sendEncodedData(final @NonNull EncodedDataImpl encodedData, @NonNull EncoderCallback callback, @NonNull Executor executor) {
            EncoderImpl.this.mEncodedDataSet.add(encodedData);
            Futures.addCallback(encodedData.getClosedFuture(), (FutureCallback)new FutureCallback<Void>(){

                public void onSuccess(@Nullable Void result) {
                    EncoderImpl.this.mEncodedDataSet.remove(encodedData);
                }

                public void onFailure(Throwable t) {
                    EncoderImpl.this.mEncodedDataSet.remove(encodedData);
                    if (t instanceof MediaCodec.CodecException) {
                        EncoderImpl.this.handleEncodeError((MediaCodec.CodecException)t);
                    } else {
                        EncoderImpl.this.handleEncodeError(0, t.getMessage(), t);
                    }
                }
            }, (Executor)EncoderImpl.this.mEncoderExecutor);
            try {
                executor.execute(() -> callback.onEncodedData(encodedData));
            }
            catch (RejectedExecutionException e) {
                Logger.e((String)EncoderImpl.this.mTag, (String)"Unable to post to the supplied executor.", (Throwable)e);
                encodedData.close();
            }
        }

        private boolean checkBufferInfo(@NonNull MediaCodec.BufferInfo bufferInfo) {
            if (this.mHasEndData) {
                Logger.d((String)EncoderImpl.this.mTag, (String)"Drop buffer by already reach end of stream.");
                return true;
            }
            if (bufferInfo.size <= 0) {
                Logger.d((String)EncoderImpl.this.mTag, (String)"Drop buffer by invalid buffer size.");
                return true;
            }
            if ((bufferInfo.flags & 2) != 0) {
                Logger.d((String)EncoderImpl.this.mTag, (String)"Drop buffer by codec config.");
                return true;
            }
            if (bufferInfo.presentationTimeUs <= this.mLastPresentationTimeUs) {
                Logger.d((String)EncoderImpl.this.mTag, (String)"Drop buffer by out of order buffer from MediaCodec.");
                return true;
            }
            this.mLastPresentationTimeUs = bufferInfo.presentationTimeUs;
            if (!EncoderImpl.this.mStartStopTimeRangeUs.contains((Comparable)Long.valueOf(bufferInfo.presentationTimeUs))) {
                Logger.d((String)EncoderImpl.this.mTag, (String)"Drop buffer by not in start-stop range.");
                if (EncoderImpl.this.mPendingCodecStop && bufferInfo.presentationTimeUs >= (Long)EncoderImpl.this.mStartStopTimeRangeUs.getUpper()) {
                    if (EncoderImpl.this.mStopTimeoutFuture != null) {
                        EncoderImpl.this.mStopTimeoutFuture.cancel(true);
                    }
                    EncoderImpl.this.mLastDataStopTimestamp = bufferInfo.presentationTimeUs;
                    EncoderImpl.this.signalCodecStop();
                    EncoderImpl.this.mPendingCodecStop = false;
                }
                return true;
            }
            if (this.updatePauseRangeStateAndCheckIfBufferPaused(bufferInfo)) {
                Logger.d((String)EncoderImpl.this.mTag, (String)"Drop buffer by pause.");
                return true;
            }
            if (EncoderImpl.this.getAdjustedTimeUs(bufferInfo) <= this.mLastSentPresentationTimeUs) {
                Logger.d((String)EncoderImpl.this.mTag, (String)"Drop buffer by adjusted time is less than the last sent time.");
                if (EncoderImpl.this.mIsVideoEncoder && EncoderImpl.isKeyFrame(bufferInfo)) {
                    this.mIsKeyFrameRequired = true;
                }
                return true;
            }
            if (!this.mHasFirstData && !this.mIsKeyFrameRequired && EncoderImpl.this.mIsVideoEncoder) {
                this.mIsKeyFrameRequired = true;
            }
            if (this.mIsKeyFrameRequired) {
                if (!EncoderImpl.isKeyFrame(bufferInfo)) {
                    Logger.d((String)EncoderImpl.this.mTag, (String)"Drop buffer by not a key frame.");
                    EncoderImpl.this.requestKeyFrameToMediaCodec();
                    return true;
                }
                this.mIsKeyFrameRequired = false;
            }
            return false;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private boolean updatePauseRangeStateAndCheckIfBufferPaused(@NonNull MediaCodec.BufferInfo bufferInfo) {
            EncoderImpl.this.updateTotalPausedDuration(bufferInfo.presentationTimeUs);
            boolean isInPauseRange = EncoderImpl.this.isInPauseRange(bufferInfo.presentationTimeUs);
            if (!this.mIsOutputBufferInPauseState && isInPauseRange) {
                EncoderCallback encoderCallback;
                Executor executor;
                Logger.d((String)EncoderImpl.this.mTag, (String)"Switch to pause state");
                this.mIsOutputBufferInPauseState = true;
                Object object = EncoderImpl.this.mLock;
                synchronized (object) {
                    executor = EncoderImpl.this.mEncoderCallbackExecutor;
                    encoderCallback = EncoderImpl.this.mEncoderCallback;
                }
                executor.execute(encoderCallback::onEncodePaused);
                if (!(EncoderImpl.this.mState != InternalState.PAUSED || !EncoderImpl.this.mIsVideoEncoder && DeviceQuirks.get(AudioEncoderIgnoresInputTimestampQuirk.class) != null || EncoderImpl.this.mIsVideoEncoder && DeviceQuirks.get(VideoEncoderSuspendDoesNotIncludeSuspendTimeQuirk.class) != null)) {
                    if (EncoderImpl.this.mEncoderInput instanceof ByteBufferInput) {
                        ((ByteBufferInput)EncoderImpl.this.mEncoderInput).setActive(false);
                    }
                    EncoderImpl.this.setMediaCodecPaused(true);
                }
                EncoderImpl.this.mLastDataStopTimestamp = bufferInfo.presentationTimeUs;
                if (EncoderImpl.this.mPendingCodecStop) {
                    if (EncoderImpl.this.mStopTimeoutFuture != null) {
                        EncoderImpl.this.mStopTimeoutFuture.cancel(true);
                    }
                    EncoderImpl.this.signalCodecStop();
                    EncoderImpl.this.mPendingCodecStop = false;
                }
            } else if (this.mIsOutputBufferInPauseState && !isInPauseRange) {
                Logger.d((String)EncoderImpl.this.mTag, (String)"Switch to resume state");
                this.mIsOutputBufferInPauseState = false;
                if (EncoderImpl.this.mIsVideoEncoder && !EncoderImpl.isKeyFrame(bufferInfo)) {
                    this.mIsKeyFrameRequired = true;
                }
            }
            return this.mIsOutputBufferInPauseState;
        }

        public void onError(@NonNull MediaCodec mediaCodec, @NonNull MediaCodec.CodecException e) {
            EncoderImpl.this.mEncoderExecutor.execute(() -> {
                switch (EncoderImpl.this.mState) {
                    case STARTED: 
                    case PAUSED: 
                    case STOPPING: 
                    case PENDING_START_PAUSED: 
                    case PENDING_START: 
                    case PENDING_RELEASE: {
                        EncoderImpl.this.handleEncodeError(e);
                        break;
                    }
                    case CONFIGURED: 
                    case ERROR: 
                    case RELEASED: {
                        break;
                    }
                    default: {
                        throw new IllegalStateException("Unknown state: " + (Object)((Object)EncoderImpl.this.mState));
                    }
                }
            });
        }

        public void onOutputFormatChanged(@NonNull MediaCodec mediaCodec, @NonNull MediaFormat mediaFormat) {
            EncoderImpl.this.mEncoderExecutor.execute(() -> {
                switch (EncoderImpl.this.mState) {
                    case STARTED: 
                    case PAUSED: 
                    case STOPPING: 
                    case PENDING_START_PAUSED: 
                    case PENDING_START: 
                    case PENDING_RELEASE: {
                        Executor executor;
                        EncoderCallback encoderCallback;
                        Object object = EncoderImpl.this.mLock;
                        synchronized (object) {
                            encoderCallback = EncoderImpl.this.mEncoderCallback;
                            executor = EncoderImpl.this.mEncoderCallbackExecutor;
                        }
                        try {
                            executor.execute(() -> encoderCallback.onOutputConfigUpdate(() -> mediaFormat));
                        }
                        catch (RejectedExecutionException e) {
                            Logger.e((String)EncoderImpl.this.mTag, (String)"Unable to post to the supplied executor.", (Throwable)e);
                        }
                        break;
                    }
                    case CONFIGURED: 
                    case ERROR: 
                    case RELEASED: {
                        break;
                    }
                    default: {
                        throw new IllegalStateException("Unknown state: " + (Object)((Object)EncoderImpl.this.mState));
                    }
                }
            });
        }
    }

    static enum InternalState {
        CONFIGURED,
        STARTED,
        PAUSED,
        STOPPING,
        PENDING_START,
        PENDING_START_PAUSED,
        PENDING_RELEASE,
        ERROR,
        RELEASED;

    }
}

