/*
 * Decompiled with CFR 0.152.
 */
package androidx.camera.video.internal;

import android.annotation.SuppressLint;
import android.content.Context;
import android.media.AudioFormat;
import android.media.AudioManager;
import android.media.AudioRecord;
import android.media.AudioRecordingConfiguration;
import android.media.AudioTimestamp;
import android.os.Build;
import androidx.annotation.IntRange;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.RequiresApi;
import androidx.annotation.RequiresPermission;
import androidx.camera.core.Logger;
import androidx.camera.core.impl.Observable;
import androidx.camera.core.impl.utils.executor.CameraXExecutors;
import androidx.camera.core.impl.utils.futures.FutureCallback;
import androidx.camera.core.impl.utils.futures.Futures;
import androidx.camera.video.internal.AudioSourceAccessException;
import androidx.camera.video.internal.AutoValue_AudioSource_Settings;
import androidx.camera.video.internal.BufferProvider;
import androidx.camera.video.internal.compat.Api23Impl;
import androidx.camera.video.internal.compat.Api24Impl;
import androidx.camera.video.internal.compat.Api29Impl;
import androidx.camera.video.internal.compat.Api31Impl;
import androidx.camera.video.internal.encoder.InputBuffer;
import androidx.core.util.Preconditions;
import com.google.auto.value.AutoValue;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

@RequiresApi(value=21)
public final class AudioSource {
    private static final String TAG = "AudioSource";
    public static final List<Integer> COMMON_SAMPLE_RATES = Collections.unmodifiableList(Arrays.asList(48000, 44100, 22050, 11025, 8000, 4800));
    final Executor mExecutor;
    private AudioManager.AudioRecordingCallback mAudioRecordingCallback;
    AtomicBoolean mSourceSilence = new AtomicBoolean(false);
    final AudioRecord mAudioRecord;
    final int mBufferSize;
    InternalState mState = InternalState.CONFIGURED;
    BufferProvider.State mBufferProviderState = BufferProvider.State.INACTIVE;
    boolean mIsSendingAudio;
    Executor mCallbackExecutor;
    AudioSourceCallback mAudioSourceCallback;
    BufferProvider<InputBuffer> mBufferProvider;
    private FutureCallback<InputBuffer> mAcquireBufferCallback;
    private Observable.Observer<BufferProvider.State> mStateObserver;

    @RequiresPermission(value="android.permission.RECORD_AUDIO")
    public AudioSource(@NonNull Settings settings, @NonNull Executor executor, @Nullable Context attributionContext) throws AudioSourceAccessException {
        if (!AudioSource.isSettingsSupported(settings.getSampleRate(), settings.getChannelCount(), settings.getAudioFormat())) {
            throw new UnsupportedOperationException(String.format("The combination of sample rate %d, channel count %d and audio format %d is not supported.", settings.getSampleRate(), settings.getChannelCount(), settings.getAudioFormat()));
        }
        int minBufferSize = AudioSource.getMinBufferSize(settings.getSampleRate(), settings.getChannelCount(), settings.getAudioFormat());
        Preconditions.checkState((minBufferSize > 0 ? 1 : 0) != 0);
        this.mExecutor = CameraXExecutors.newSequentialExecutor((Executor)executor);
        this.mBufferSize = minBufferSize * 2;
        try {
            if (Build.VERSION.SDK_INT >= 23) {
                AudioFormat audioFormatObj = new AudioFormat.Builder().setSampleRate(settings.getSampleRate()).setChannelMask(AudioSource.channelCountToChannelMask(settings.getChannelCount())).setEncoding(settings.getAudioFormat()).build();
                AudioRecord.Builder audioRecordBuilder = Api23Impl.createAudioRecordBuilder();
                if (Build.VERSION.SDK_INT >= 31 && attributionContext != null) {
                    Api31Impl.setContext(audioRecordBuilder, attributionContext);
                }
                Api23Impl.setAudioSource(audioRecordBuilder, settings.getAudioSource());
                Api23Impl.setAudioFormat(audioRecordBuilder, audioFormatObj);
                Api23Impl.setBufferSizeInBytes(audioRecordBuilder, this.mBufferSize);
                this.mAudioRecord = Api23Impl.build(audioRecordBuilder);
            } else {
                this.mAudioRecord = new AudioRecord(settings.getAudioSource(), settings.getSampleRate(), AudioSource.channelCountToChannelConfig(settings.getChannelCount()), settings.getAudioFormat(), this.mBufferSize);
            }
        }
        catch (IllegalArgumentException e) {
            throw new AudioSourceAccessException("Unable to create AudioRecord", e);
        }
        if (this.mAudioRecord.getState() != 1) {
            this.mAudioRecord.release();
            throw new AudioSourceAccessException("Unable to initialize AudioRecord");
        }
        if (Build.VERSION.SDK_INT >= 29) {
            this.mAudioRecordingCallback = new AudioRecordingApi29Callback();
            Api29Impl.registerAudioRecordingCallback(this.mAudioRecord, this.mExecutor, this.mAudioRecordingCallback);
        }
    }

    public void setBufferProvider(@NonNull BufferProvider<InputBuffer> bufferProvider) {
        this.mExecutor.execute(() -> {
            switch (this.mState) {
                case CONFIGURED: 
                case STARTED: {
                    if (this.mBufferProvider == bufferProvider) break;
                    this.resetBufferProvider(bufferProvider);
                    break;
                }
                case RELEASED: {
                    throw new IllegalStateException("AudioRecorder is released");
                }
            }
        });
    }

    public void start() {
        this.mExecutor.execute(() -> {
            switch (this.mState) {
                case CONFIGURED: {
                    this.setState(InternalState.STARTED);
                    this.updateSendingAudio();
                    break;
                }
                case STARTED: {
                    break;
                }
                case RELEASED: {
                    throw new IllegalStateException("AudioRecorder is released");
                }
            }
        });
    }

    public void stop() {
        this.mExecutor.execute(() -> {
            switch (this.mState) {
                case STARTED: {
                    this.setState(InternalState.CONFIGURED);
                    this.updateSendingAudio();
                    break;
                }
                case CONFIGURED: {
                    break;
                }
                case RELEASED: {
                    throw new IllegalStateException("AudioRecorder is released");
                }
            }
        });
    }

    public void release() {
        this.mExecutor.execute(() -> {
            switch (this.mState) {
                case CONFIGURED: 
                case STARTED: {
                    this.resetBufferProvider(null);
                    if (Build.VERSION.SDK_INT >= 29) {
                        Api29Impl.unregisterAudioRecordingCallback(this.mAudioRecord, this.mAudioRecordingCallback);
                    }
                    this.mAudioRecord.release();
                    this.stopSendingAudio();
                    this.setState(InternalState.RELEASED);
                    break;
                }
            }
        });
    }

    public void setAudioSourceCallback(@NonNull Executor executor, @NonNull AudioSourceCallback callback) {
        this.mExecutor.execute(() -> {
            switch (this.mState) {
                case CONFIGURED: {
                    this.mCallbackExecutor = executor;
                    this.mAudioSourceCallback = callback;
                    break;
                }
                case STARTED: 
                case RELEASED: {
                    throw new IllegalStateException("The audio recording callback must be registered before the audio source is started.");
                }
            }
        });
    }

    private void resetBufferProvider(final @Nullable BufferProvider<InputBuffer> bufferProvider) {
        if (this.mBufferProvider != null) {
            this.mBufferProvider.removeObserver(this.mStateObserver);
            this.mBufferProvider = null;
            this.mStateObserver = null;
            this.mAcquireBufferCallback = null;
        }
        this.mBufferProviderState = BufferProvider.State.INACTIVE;
        this.updateSendingAudio();
        if (bufferProvider != null) {
            this.mBufferProvider = bufferProvider;
            this.mStateObserver = new Observable.Observer<BufferProvider.State>(){

                public void onNewData(@Nullable BufferProvider.State state) {
                    if (AudioSource.this.mBufferProvider == bufferProvider) {
                        Logger.d((String)AudioSource.TAG, (String)("Receive BufferProvider state change: " + (Object)((Object)AudioSource.this.mBufferProviderState) + " to " + (Object)((Object)state)));
                        AudioSource.this.mBufferProviderState = state;
                        AudioSource.this.updateSendingAudio();
                    }
                }

                public void onError(@NonNull Throwable throwable) {
                    if (AudioSource.this.mBufferProvider == bufferProvider) {
                        AudioSource.this.notifyError(throwable);
                    }
                }
            };
            this.mAcquireBufferCallback = new FutureCallback<InputBuffer>(){

                public void onSuccess(InputBuffer inputBuffer) {
                    if (!AudioSource.this.mIsSendingAudio || AudioSource.this.mBufferProvider != bufferProvider) {
                        inputBuffer.cancel();
                        return;
                    }
                    ByteBuffer byteBuffer = inputBuffer.getByteBuffer();
                    int length = AudioSource.this.mAudioRecord.read(byteBuffer, AudioSource.this.mBufferSize);
                    if (length > 0) {
                        byteBuffer.limit(length);
                        inputBuffer.setPresentationTimeUs(AudioSource.this.generatePresentationTimeUs());
                        inputBuffer.submit();
                    } else {
                        Logger.w((String)AudioSource.TAG, (String)"Unable to read data from AudioRecord.");
                        inputBuffer.cancel();
                    }
                    AudioSource.this.sendNextAudio();
                }

                public void onFailure(Throwable throwable) {
                    if (AudioSource.this.mBufferProvider != bufferProvider) {
                        Logger.d((String)AudioSource.TAG, (String)"Unable to get input buffer, the BufferProvider could be transitioning to INACTIVE state.");
                        AudioSource.this.notifyError(throwable);
                    }
                }
            };
            this.mBufferProvider.addObserver(this.mExecutor, this.mStateObserver);
        }
    }

    void notifyError(Throwable throwable) {
        if (this.mCallbackExecutor != null && this.mAudioSourceCallback != null) {
            this.mCallbackExecutor.execute(() -> this.mAudioSourceCallback.onError(throwable));
        }
    }

    void updateSendingAudio() {
        if (this.mState == InternalState.STARTED && this.mBufferProviderState == BufferProvider.State.ACTIVE) {
            this.startSendingAudio();
        } else {
            this.stopSendingAudio();
        }
    }

    private void startSendingAudio() {
        if (this.mIsSendingAudio) {
            return;
        }
        try {
            Logger.d((String)TAG, (String)"startSendingAudio");
            this.mAudioRecord.startRecording();
            if (this.mAudioRecord.getRecordingState() != 3) {
                throw new IllegalStateException("Unable to start AudioRecord with state: " + this.mAudioRecord.getRecordingState());
            }
        }
        catch (IllegalStateException e) {
            Logger.w((String)TAG, (String)"Failed to start AudioRecord", (Throwable)e);
            this.setState(InternalState.CONFIGURED);
            this.notifyError(new AudioSourceAccessException("Unable to start the audio record.", e));
            return;
        }
        this.mIsSendingAudio = true;
        this.sendNextAudio();
    }

    private void stopSendingAudio() {
        if (!this.mIsSendingAudio) {
            return;
        }
        this.mIsSendingAudio = false;
        try {
            Logger.d((String)TAG, (String)"stopSendingAudio");
            this.mAudioRecord.stop();
            if (this.mAudioRecord.getRecordingState() != 1) {
                throw new IllegalStateException("Unable to stop AudioRecord with state: " + this.mAudioRecord.getRecordingState());
            }
        }
        catch (IllegalStateException e) {
            Logger.w((String)TAG, (String)"Failed to stop AudioRecord", (Throwable)e);
            this.notifyError(e);
        }
    }

    void sendNextAudio() {
        Futures.addCallback(this.mBufferProvider.acquireBuffer(), this.mAcquireBufferCallback, (Executor)this.mExecutor);
    }

    void setState(InternalState state) {
        Logger.d((String)TAG, (String)("Transitioning internal state: " + (Object)((Object)this.mState) + " --> " + (Object)((Object)state)));
        this.mState = state;
    }

    long generatePresentationTimeUs() {
        long presentationTimeUs = -1L;
        if (Build.VERSION.SDK_INT >= 24) {
            AudioTimestamp audioTimestamp = new AudioTimestamp();
            if (Api24Impl.getTimestamp(this.mAudioRecord, audioTimestamp, 0) == 0) {
                presentationTimeUs = TimeUnit.NANOSECONDS.toMicros(audioTimestamp.nanoTime);
            } else {
                Logger.w((String)TAG, (String)"Unable to get audio timestamp");
            }
        }
        if (presentationTimeUs == -1L) {
            presentationTimeUs = TimeUnit.NANOSECONDS.toMicros(System.nanoTime());
        }
        return presentationTimeUs;
    }

    public static boolean isSettingsSupported(int sampleRate, int channelCount, int audioFormat) {
        if (sampleRate <= 0 || channelCount <= 0) {
            return false;
        }
        return AudioSource.getMinBufferSize(sampleRate, channelCount, audioFormat) > 0;
    }

    private static int channelCountToChannelConfig(int channelCount) {
        return channelCount == 1 ? 16 : 12;
    }

    private static int channelCountToChannelMask(int channelCount) {
        return channelCount == 1 ? 16 : 12;
    }

    private static int getMinBufferSize(int sampleRate, int channelCount, int audioFormat) {
        return AudioRecord.getMinBufferSize((int)sampleRate, (int)AudioSource.channelCountToChannelConfig(channelCount), (int)audioFormat);
    }

    public static interface AudioSourceCallback {
        public void onSilenced(boolean var1);

        public void onError(@NonNull Throwable var1);
    }

    @AutoValue
    public static abstract class Settings {
        @SuppressLint(value={"Range"})
        @NonNull
        public static Builder builder() {
            return new AutoValue_AudioSource_Settings.Builder().setAudioSource(-1).setSampleRate(-1).setChannelCount(-1).setAudioFormat(-1);
        }

        @NonNull
        public abstract Builder toBuilder();

        public abstract int getAudioSource();

        @IntRange(from=1L)
        public abstract int getSampleRate();

        @IntRange(from=1L)
        public abstract int getChannelCount();

        public abstract int getAudioFormat();

        Settings() {
        }

        @AutoValue.Builder
        public static abstract class Builder {
            @NonNull
            public abstract Builder setAudioSource(int var1);

            @NonNull
            public abstract Builder setSampleRate(@IntRange(from=1L) int var1);

            @NonNull
            public abstract Builder setChannelCount(@IntRange(from=1L) int var1);

            @NonNull
            public abstract Builder setAudioFormat(int var1);

            abstract Settings autoBuild();

            @NonNull
            public final Settings build() {
                Settings settings = this.autoBuild();
                String missingOrInvalid = "";
                if (settings.getAudioSource() == -1) {
                    missingOrInvalid = missingOrInvalid + " audioSource";
                }
                if (settings.getSampleRate() <= 0) {
                    missingOrInvalid = missingOrInvalid + " sampleRate";
                }
                if (settings.getChannelCount() <= 0) {
                    missingOrInvalid = missingOrInvalid + " channelCount";
                }
                if (settings.getAudioFormat() == -1) {
                    missingOrInvalid = missingOrInvalid + " audioFormat";
                }
                if (!missingOrInvalid.isEmpty()) {
                    throw new IllegalArgumentException("Required settings missing or non-positive:" + missingOrInvalid);
                }
                return settings;
            }

            Builder() {
            }
        }
    }

    @RequiresApi(value=29)
    class AudioRecordingApi29Callback
    extends AudioManager.AudioRecordingCallback {
        AudioRecordingApi29Callback() {
        }

        public void onRecordingConfigChanged(List<AudioRecordingConfiguration> configs) {
            super.onRecordingConfigChanged(configs);
            if (AudioSource.this.mCallbackExecutor != null && AudioSource.this.mAudioSourceCallback != null) {
                for (AudioRecordingConfiguration config : configs) {
                    if (Api24Impl.getClientAudioSessionId(config) != AudioSource.this.mAudioRecord.getAudioSessionId()) continue;
                    boolean isSilenced = Api29Impl.isClientSilenced(config);
                    if (AudioSource.this.mSourceSilence.getAndSet(isSilenced) == isSilenced) break;
                    AudioSource.this.mCallbackExecutor.execute(() -> AudioSource.this.mAudioSourceCallback.onSilenced(isSilenced));
                    break;
                }
            }
        }
    }

    static enum InternalState {
        CONFIGURED,
        STARTED,
        RELEASED;

    }
}

