package com.particles.mes.android.data.mapper

import com.google.protobuf.ByteString
import com.particles.mes.android.data.MesAdHideEvent
import com.particles.mes.android.data.MesAdImpressionEvent
import com.particles.mes.android.data.MesAdReportEvent
import com.particles.mes.android.data.MesAdRequest
import com.particles.mes.android.data.MesAdRequestExt
import com.particles.mes.android.data.MesAdResponse
import com.particles.mes.android.data.MesAdType
import com.particles.mes.protos.AdHideEvents
import com.particles.mes.protos.AdImpressionEvents.AdImpressionEvent
import com.particles.mes.protos.AdReportEvents
import com.particles.mes.protos.Common
import com.particles.mes.protos.Common.Ad
import com.particles.mes.protos.Common.AdType
import com.particles.mes.protos.Common.RequestContext
import com.particles.mes.protos.Common.RequestContextExt
import com.particles.mes.protos.openrtb.BidResponse
import com.particles.msp.MSPManager
import org.json.JSONException
import org.json.JSONObject
import org.prebid.mobile.rendering.bidding.data.bid.Bid

object EventMapper {
    fun map(source: MesAdImpressionEvent): AdImpressionEvent {
        return AdImpressionEvent.newBuilder()
            .apply {
                tsMs = source.ts
                requestContext = map(source.adRequest)
                ad = map(source.adResponse)
                os = Common.OsType.OS_TYPE_ANDROID
                org = MSPManager.org
                app = MSPManager.app
            }
            .build()
    }

    fun map(source: MesAdHideEvent): AdHideEvents.AdHideEvent {
        return AdHideEvents.AdHideEvent.newBuilder()
            .apply {
                tsMs = source.ts
                requestContext = map(source.adRequest)
                ad = map(source.adResponse)
                os = Common.OsType.OS_TYPE_ANDROID
                reason = source.reason
                org = MSPManager.org
                app = MSPManager.app
            }
            .build()
    }

    fun map(source: MesAdReportEvent): AdReportEvents.AdReportEvent {
        return AdReportEvents.AdReportEvent.newBuilder()
            .apply {
                tsMs = source.ts
                requestContext = map(source.adRequest)
                ad = map(source.adResponse)
                reason = source.reason
                os = Common.OsType.OS_TYPE_ANDROID
                org = MSPManager.org
                app = MSPManager.app
                source.description?.let { description = it }
            }
            .build()
    }

    private fun map(source: MesAdRequest): RequestContext {
        return RequestContext.newBuilder()
            .apply {
                tsMs = source.sentRequestAtMillis
                source.bidRequest?.let { bidRequest = BidRequestMapper.map(it) }
                isOpenRtbRequest = source.bidRequest != null
                ext = map(source.ext)
            }
            .build()
    }

    private fun map(source: MesAdRequestExt): RequestContextExt {
        return RequestContextExt.newBuilder()
            .apply {
                source.docId?.let { docId = it }
                source.sessionId?.let { sessionId = it }
                source.source?.let { this.source = it }
                position = source.position
                placementId = source.placementId
                source.buckets.forEach { addBuckets(it) }
                adSlotId = source.adSlotId
                userId = source.userId
            }
            .build()
    }

    private fun map(source: MesAdResponse): Ad {
        return Ad.newBuilder()
            .apply {
                tsMs = source.receivedResponseAtMillis
                type = map(source.adType)
                source.bid?.let {
                    seatBid = BidResponse.SeatBid.newBuilder()
                        .setSeat(getSeat(it))
                        .addBid(BidResponseMapper.map(it))
                        .build()
                }
                source.title?.let { title = it }
                source.body?.let { body = it }
                source.advertiser?.let { advertiser = it }
                source.adScreenshot?.let { adScreenshot = ByteString.copyFrom(it) }
                source.fullScreenshot?.let { fullScreenshot = ByteString.copyFrom(it) }
            }
            .build()
    }

    private fun getSeat(bid: Bid): String {
        return try {
            JSONObject(bid.jsonString).getJSONObject("ext")
                .getJSONObject("prebid")
                .getJSONObject("meta")
                .getString("adaptercode")
        } catch (ignored: JSONException) {
            ""
        }
    }

    private fun map(source: MesAdType): AdType {
        return when (source) {
            MesAdType.AD_TYPE_UNSPECIFIED -> AdType.AD_TYPE_UNSPECIFIED
            MesAdType.AD_TYPE_NATIVE -> AdType.AD_TYPE_NATIVE
            MesAdType.AD_TYPE_DISPLAY -> AdType.AD_TYPE_DISPLAY
            MesAdType.AD_TYPE_INTERSTITIAL -> AdType.AD_TYPE_INTERSTITIAL
            MesAdType.AD_TYPE_VIDEO -> AdType.AD_TYPE_VIDEO
        }
    }
}
