package com.particles.mes.android.data.mapper

import com.google.protobuf.ByteString
import com.particles.mes.android.data.AppSignal
import com.particles.mes.android.data.ConnectionType
import com.particles.mes.android.data.DeviceSignal
import com.particles.mes.android.data.MesAdClickEvent
import com.particles.mes.android.data.MesAdHideEvent
import com.particles.mes.android.data.MesAdImpressionEvent
import com.particles.mes.android.data.MesAdReportEvent
import com.particles.mes.android.data.MesAdRequest
import com.particles.mes.android.data.MesAdRequestEvent
import com.particles.mes.android.data.MesAdRequestExt
import com.particles.mes.android.data.MesAdResponse
import com.particles.mes.android.data.MesAdResponseEvent
import com.particles.mes.android.data.MesAdType
import com.particles.mes.android.data.MesGetAdEvent
import com.particles.mes.android.data.MesLoadAdEvent
import com.particles.mes.android.data.MesSdkInitEvent
import com.particles.mes.android.data.MesUserSignalEvent
import com.particles.mes.android.data.SdkSignal
import com.particles.mes.protos.Ad
import com.particles.mes.protos.AdClickEvent
import com.particles.mes.protos.AdHideEvent
import com.particles.mes.protos.AdImpressionEvent
import com.particles.mes.protos.AdReportEvent
import com.particles.mes.protos.AdRequest
import com.particles.mes.protos.AdResponse
import com.particles.mes.protos.AdType
import com.particles.mes.protos.GetAdEvent
import com.particles.mes.protos.LoadAd
import com.particles.mes.protos.OsType
import com.particles.mes.protos.RequestContext
import com.particles.mes.protos.RequestContextExt
import com.particles.mes.protos.SDKSignal
import com.particles.mes.protos.SdkInitEvent
import com.particles.mes.protos.SdkPlatform
import com.particles.mes.protos.UserSignal
import com.particles.mes.protos.openrtb.BidResponse
import org.json.JSONException
import org.json.JSONObject
import org.prebid.mobile.rendering.bidding.data.bid.Bid

object EventMapper {
    fun map(source: MesAdClickEvent): AdClickEvent {
        return AdClickEvent.newBuilder()
            .apply {
                tsMs = source.ts
                requestContext = map(source.adRequest)
                ad = map(source.adResponse)
                os = OsType.OS_TYPE_ANDROID
                org = source.adRequest.org
                app = source.adRequest.app
                mspSdkVersion = source.mspSdkVersion
            }
            .build()
    }

    fun map(source: MesAdImpressionEvent): AdImpressionEvent {
        return AdImpressionEvent.newBuilder()
            .apply {
                tsMs = source.ts
                requestContext = map(source.adRequest)
                ad = map(source.adResponse)
                os = OsType.OS_TYPE_ANDROID
                org = source.adRequest.org
                app = source.adRequest.app
                mspSdkVersion = source.mspSdkVersion
            }
            .build()
    }

    fun map(source: MesUserSignalEvent): UserSignal {
        return UserSignal.newBuilder()
            .apply {
                type = when (source.type) {
                    com.particles.mes.android.data.UserSignalType.USER_SIGNAL_TYPE_INTO_BACKGROUND -> com.particles.mes.protos.UserSignalType.USER_SIGNAL_TYPE_INTO_BACKGROUND
                    com.particles.mes.android.data.UserSignalType.USER_SIGNAL_TYPE_INTO_FOREGROUND -> com.particles.mes.protos.UserSignalType.USER_SIGNAL_TYPE_INTO_FOREGROUND
                    com.particles.mes.android.data.UserSignalType.USER_SIGNAL_TYPE_INTO_ATTRIBUTION -> com.particles.mes.protos.UserSignalType.USER_SIGNAL_TYPE_ATTRIBUTION
                }
                sdkSignal = map(source.sdkSignal)
                deviceSignal = map(source.deviceSignal)
                appSignal = map(source.appSignal)
            }
            .build()
    }

    fun map(source: MesSdkInitEvent): SdkInitEvent {
        return SdkInitEvent.newBuilder()
            .apply {
                clientTsMs = source.clientTs
                os = OsType.OS_TYPE_ANDROID
                org = source.org
                app = source.app
                latency = source.latency
                totalCompleteTime = source.totalCompleteTime
                putAllCompleteTimeByAdNetwork(source.completeTimeByAdNetwork)
                mspSdkVersion = source.mspSdkVersion
                mspId = source.mspId
                ifa = source.ifa
                batteryLevel = source.batteryLevel
                batteryStatus = source.batteryStatus
                fontSize = source.fontSize
                timezone = source.timeZone
                availableMemoryBytes = source.availableMemoryBytes
                isLowPowerMode = source.isLowPowerMode
                isLowDataMode = source.isLowDataMode
                ppid = source.ppid
                sdkSignal = map(source.sdkSignal)
                deviceSignal = map(source.deviceSignal)
                appSignal = map(source.appSignal)
            }
            .build()
    }

    fun map(source: MesLoadAdEvent): LoadAd {
        return LoadAd.newBuilder()
            .apply {
                clientTsMs = source.clientTs
                source.adRequest?.let { requestContext = map(it) }
                source.adResponse?.let { ad = map(it) }
                os = OsType.OS_TYPE_ANDROID
                org = source.org
                app = source.app
                latency = source.latency
                errorCode = source.errorCode
                source.errorMessage?.let { errorMessage = it }
                filledFromCache = source.filledFromCache
                mspSdkVersion = source.mspSdkVersion
            }
            .build()
    }

    fun map(source: SdkSignal): SDKSignal {
        return SDKSignal.newBuilder()
            .apply {
                orgId = source.orgId
                appId = source.appId
                mspId = source.mspId
                sdkVersion = source.sdkVersion
                platform = SdkPlatform.SDK_PLATFORM_ANDROID
                clientTs = source.clientTs
                uuid = source.uuid
            }
            .build()
    }

    fun map(source: AppSignal): com.particles.mes.protos.AppSignal {
        return com.particles.mes.protos.AppSignal.newBuilder()
            .apply {
                name = source.name
                bundle = source.bundle
                ver = source.ver
                domain = source.domain
                page = source.page
                ppid = source.ppid
            }
            .build()
    }

    fun map(source: DeviceSignal): com.particles.mes.protos.DeviceSignal {
        return com.particles.mes.protos.DeviceSignal.newBuilder()
            .apply {
                make = source.make
                model = source.model
                os = source.os
                osv = source.osv
                kernelv = source.kernelv
                w = source.w
                h = source.h
                orientation = source.orientation
                volumeLevel = source.volumeLevel
                brightness = source.brightness
                batteryLevel = source.batteryLevel
                batteryStatus = source.batteryStatus
                thermalStatus = source.thermalStatus
                totalMemoryBytes = source.totalMemoryBytes
                availableMemoryBytes = source.availableMemoryBytes
                totalStorageBytes = source.totalStorageBytes
                availableStorageBytes = source.availableStorageBytes
                isLowPowerMode = source.isLowPowerMode
                isLowDataMode = source.isLowDataMode
                fontSize = source.fontSize
                theme = source.theme
                carrier = source.carrier
                mccmnc = source.mccmnc
                connectionType = when (source.connectionType) {
                    ConnectionType.CONNECTION_TYPE_WIFI -> com.particles.mes.protos.ConnectionType.CONNECTION_TYPE_WIFI
                    ConnectionType.CONNECTION_TYPE_CELL_UNKNOWN -> com.particles.mes.protos.ConnectionType.CONNECTION_TYPE_CELL_UNKNOWN
                    ConnectionType.CONNECTION_TYPE_CELL_2G -> com.particles.mes.protos.ConnectionType.CONNECTION_TYPE_CELL_2G
                    ConnectionType.CONNECTION_TYPE_CELL_3G -> com.particles.mes.protos.ConnectionType.CONNECTION_TYPE_CELL_3G
                    ConnectionType.CONNECTION_TYPE_CELL_4G -> com.particles.mes.protos.ConnectionType.CONNECTION_TYPE_CELL_4G
                    ConnectionType.CONNECTION_TYPE_CELL_5G -> com.particles.mes.protos.ConnectionType.CONNECTION_TYPE_CELL_5G
                    ConnectionType.CONNECTION_TYPE_ETHERNET -> com.particles.mes.protos.ConnectionType.CONNECTION_TYPE_ETHERNET
                    else -> com.particles.mes.protos.ConnectionType.CONNECTION_TYPE_UNSPECIFIED
                }
                country = source.country
                timezone = source.timezone
                locale = source.locale
                ua = source.ua
                ifa = source.ifa
                ifv = source.ifv
                lmt = source.lmt
            }
            .build()
    }

    fun map(source: MesGetAdEvent): GetAdEvent {
        return GetAdEvent.newBuilder()
            .apply {
                clientTsMs = source.clientTs
                requestContext = map(source.adRequest)
                ad = map(source.adResponse)
                os = OsType.OS_TYPE_ANDROID
                org = source.adRequest.org
                app = source.adRequest.app
                errorCode = source.errorCode
                source.errorMessage?.let { errorMessage = it }
                mspSdkVersion = source.mspSdkVersion
            }
            .build()
    }

    fun map(source: MesAdRequestEvent): AdRequest {
        return AdRequest.newBuilder()
            .apply {
                clientTsMs = source.clientTs
                requestContext = map(source.adRequest)
                os = OsType.OS_TYPE_ANDROID
                org = source.adRequest.org
                app = source.adRequest.app
                mspSdkVersion = source.mspSdkVersion
            }
            .build()
    }

    fun map(source: MesAdResponseEvent): AdResponse {
        return AdResponse.newBuilder()
            .apply {
                clientTsMs = source.clientTs
                requestContext = map(source.adRequest)
                os = OsType.OS_TYPE_ANDROID
                org = source.adRequest.org
                app = source.adRequest.app
                source.adResponse?.let {ad = map(it) }
                latency = source.latency
                errorCode = source.errorCode
                source.errorMessage?.let { errorMessage = it }
                mspSdkVersion = source.mspSdkVersion
            }
            .build()
    }

    fun map(source: MesAdHideEvent): AdHideEvent {
        return AdHideEvent.newBuilder()
            .apply {
                tsMs = source.ts
                requestContext = map(source.adRequest)
                ad = map(source.adResponse)
                os = OsType.OS_TYPE_ANDROID
                reason = source.reason
                org = source.adRequest.org
                app = source.adRequest.app
                mspSdkVersion = source.mspSdkVersion
            }
            .build()
    }

    fun map(source: MesAdReportEvent): AdReportEvent {
        return AdReportEvent.newBuilder()
            .apply {
                tsMs = source.ts
                requestContext = map(source.adRequest)
                ad = map(source.adResponse)
                reason = source.reason
                os = OsType.OS_TYPE_ANDROID
                org = source.adRequest.org
                app = source.adRequest.app
                source.description?.let { description = it }
                mspSdkVersion = source.mspSdkVersion
            }
            .build()
    }

    private fun map(source: MesAdRequest): RequestContext {
        return RequestContext.newBuilder()
            .apply {
                tsMs = source.sentRequestAtMillis
                source.bidRequest?.let { bidRequest = BidRequestMapper.map(it, source.isTest) }
                isOpenRtbRequest = source.isOpenRtb
                ext = map(source.ext)
            }
            .build()
    }

    private fun map(source: MesAdRequestExt): RequestContextExt {
        return RequestContextExt.newBuilder()
            .apply {
                source.docId?.let { docId = it }
                source.sessionId?.let { sessionId = it }
                source.source?.let { this.source = it }
                position = source.position
                placementId = source.placementId
                source.buckets.forEach { addBuckets(it) }
                adSlotId = source.adSlotId
                userId = source.userId
            }
            .build()
    }

    private fun map(source: MesAdResponse): Ad {
        return Ad.newBuilder()
            .apply {
                tsMs = source.receivedResponseAtMillis
                type = map(source.adType)
                source.bid?.let {
                    seatBid = BidResponse.SeatBid.newBuilder()
                        .setSeat(getSeat(it))
                        .addBid(BidResponseMapper.map(it))
                        .build()
                }
                source.title?.let { title = it }
                source.body?.let { body = it }
                source.advertiser?.let { advertiser = it }
                source.adScreenshot?.let { adScreenshot = ByteString.copyFrom(it) }
                source.fullScreenshot?.let { fullScreenshot = ByteString.copyFrom(it) }
            }
            .build()
    }

    private fun getSeat(bid: Bid): String {
        return try {
            JSONObject(bid.jsonString).getJSONObject("ext")
                .getJSONObject("prebid")
                .getJSONObject("meta")
                .getString("adaptercode")
        } catch (ignored: JSONException) {
            ""
        }
    }

    private fun map(source: MesAdType): AdType {
        return when (source) {
            MesAdType.AD_TYPE_UNSPECIFIED -> AdType.AD_TYPE_UNSPECIFIED
            MesAdType.AD_TYPE_NATIVE -> AdType.AD_TYPE_NATIVE
            MesAdType.AD_TYPE_DISPLAY -> AdType.AD_TYPE_DISPLAY
            MesAdType.AD_TYPE_INTERSTITIAL -> AdType.AD_TYPE_INTERSTITIAL
            MesAdType.AD_TYPE_VIDEO -> AdType.AD_TYPE_VIDEO
        }
    }
}
