package ai.connectif.sdk.model

import ai.connectif.sdk.Utils.ensurePositive
import ai.connectif.sdk.Utils.requireValidUrl
import ai.connectif.sdk.Utils.truncate
import ai.connectif.sdk.data.model.event.ProductBasketItemInternal
import java.math.BigDecimal
import java.util.Date

/**
 * Represents a product item in the user's cart.
 *
 * @property productDetailUrl Product page URL (URL must be unencoded).
 * @property productId Unique product identifier (in the eCommerce system).
 * @property name Product name.
 * @property unitPrice Product unit price.
 * @property description (Optional) Product description. It can contain HTML code.
 * @property imageUrl (Optional) Product image URL.
 * @property categories (Optional) Categories the product belongs to.
 *  If the eCommerce allows subcategories, you can indicate them with the full category path,
 *  separating each level with the '/' character.
 *  For example: "/Computers/Keyboards/Logitech".
 *  the full path (e.g. "/Computers/Keyboards/Logitech").
 * @property tags (Optional) Tags associated with the product.
 * @property brand (Optional) Product brand.
 * @property reviewCount (Optional) Number of product reviews.
 * @property rating (Optional) Product rating on a scale of 0 to 5.
 * @property ratingCount (Optional) Number of product ratings.
 * @property thumbnailUrl (Optional) Product thumbnail image URL.
 * @property unitPriceOriginal (Optional) Original product unit price before any discount.
 * @property unitPriceWithoutVAT (Optional) Product unit price without VAT.
 * @property discountedAmount (Optional) Discounted amount for the product.
 * @property discountedPercentage (Optional) Discount percentage (0–100, no '%' symbol).
 * @property publishedAt (Optional) Date the product was published in the store catalog.
 * @property customField1 (Optional) Custom product field.
 * @property customField2 (Optional) Custom product field.
 * @property customField3 (Optional) Custom product field.
 * @property quantity Quantity of units of this product.
 * @property price Total price of the products represented by this product basket item.
 *  Usually `quantity * unitPrice`.
 */
data class ProductBasketItem(
    val productDetailUrl: String,
    val productId: String,
    val name: String,
    val unitPrice: BigDecimal,
    val description: String? = null,
    val imageUrl: String? = null,
    val categories: List<String>? = null,
    val tags: List<String>? = null,
    val brand: String? = null,
    val reviewCount: Int? = null,
    val rating: Double? = null,
    val ratingCount: Int? = null,
    val thumbnailUrl: String? = null,
    val unitPriceOriginal: BigDecimal? = null,
    val unitPriceWithoutVAT: BigDecimal? = null,
    val discountedAmount: BigDecimal? = null,
    val discountedPercentage: Double? = null,
    val publishedAt: Date? = null,
    val customField1: String? = null,
    val customField2: String? = null,
    val customField3: String? = null,
    val quantity: Int,
    val price: BigDecimal
) {
    fun mapToProductBasketItemInternal(): ProductBasketItemInternal {
        require(this.quantity >= 1) { "quantity should be greater than or equal to 1" }
        require(this.name != "") { "name cannot be empty" }
        require(this.productId != "") { "productId cannot be empty" }
        require(this.productId.length <= 50) { "productId cannot have a length greater than 50" }
        require(this.productDetailUrl != "") { "productDetailUrl cannot be empty" }
        requireValidUrl(this.productDetailUrl, "productDetailUrl")
        requireValidUrl(this.imageUrl, "imageUrl")
        requireValidUrl(this.thumbnailUrl, "thumbnailUrl")

        return ProductBasketItemInternal(
            productDetailUrl = productDetailUrl,
            productId = productId,
            name = truncate(this.name, 50).toString(),
            unitPrice = unitPrice,
            description = truncate(this.description, 1000),
            imageUrl = imageUrl,
            categories = truncate(this.categories, 20, "categories")?.map {
                truncate(it, 50).toString()
            },
            tags = truncate(this.tags, 20, "tags")?.map {
                truncate(it, 50).toString()
            },
            brand = truncate(this.brand, 50),
            reviewCount = ensurePositive(this.reviewCount, "reviewCount"),
            rating = ensurePositive(this.rating, "rating"),
            ratingCount = ensurePositive(this.ratingCount, "ratingCount"),
            thumbnailUrl = thumbnailUrl,
            unitPriceOriginal = unitPriceOriginal,
            unitPriceWithoutVAT = unitPriceWithoutVAT,
            discountedAmount = discountedAmount,
            discountedPercentage = discountedPercentage,
            publishedAt = publishedAt,
            customField1 = truncate(this.customField1, 1000),
            customField2 = truncate(this.customField2, 1000),
            customField3 = truncate(this.customField3, 1000),
            quantity = quantity,
            price = price
        )
    }
}
