package ai.connectif.sdk

import ai.connectif.sdk.data.error.ConnectifApiException
import ai.connectif.sdk.data.error.UnauthorizedApiException
import ai.connectif.sdk.data.error.ValidationException
import ai.connectif.sdk.data.model.push.TrackPushClickData
import ai.connectif.sdk.data.source.LocalPushTokenDataSourceImpl
import ai.connectif.sdk.dependency.ConnectifComponent
import ai.connectif.sdk.dependency.ConnectifComponentFactory
import ai.connectif.sdk.dependency.ProductionComponentFactory
import ai.connectif.sdk.manager.Logger
import ai.connectif.sdk.model.Cart
import ai.connectif.sdk.model.ContactInfo
import ai.connectif.sdk.model.EventCallbacks
import ai.connectif.sdk.model.Login
import ai.connectif.sdk.model.PageVisit
import ai.connectif.sdk.model.Product
import ai.connectif.sdk.model.ProductVisit
import ai.connectif.sdk.model.Purchase
import ai.connectif.sdk.model.Register
import ai.connectif.sdk.model.RegisterContactInfo
import ai.connectif.sdk.model.Search
import ai.connectif.sdk.model.customevent.CustomEvent
import android.content.Context
import android.content.SharedPreferences
import java.util.Date
import java.util.concurrent.Executors
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.asCoroutineDispatcher
import kotlinx.coroutines.launch
import kotlinx.coroutines.sync.Mutex
import kotlinx.coroutines.sync.withLock
import kotlinx.coroutines.withContext

class ConnectifInternal(
    private val dispatcher: CoroutineDispatcher = Executors.newSingleThreadExecutor()
        .asCoroutineDispatcher()
) : ConnectifInterface {
    internal var componentFactory: ConnectifComponentFactory = ProductionComponentFactory()
    internal lateinit var connectifComponent: ConnectifComponent
    internal var isInitialized: Boolean = false
    private val mutex = Mutex()

    private fun runSequentially(action: suspend () -> Unit) {
        CoroutineScope(dispatcher).launch {
            mutex.withLock {
                action()
            }
        }
    }

    override fun initialize(
        context: Context,
        apiKey: String,
        email: String?,
        connectifConfig: ConnectifConfig
    ) {
        runSequentially {
            try {
                Logger.logLevel = connectifConfig.logLevel
                if (isInitialized) {
                    Logger.w("Connectif is already initialized")
                    return@runSequentially
                }
                connectifComponent =
                    componentFactory.create(context, apiKey)
                connectifComponent.settingRepository.config = connectifConfig
                email?.let {
                    if (Utils.isValidEmail(it)) {
                        connectifComponent.settingRepository.email = it
                    } else {
                        Logger.w("Connectif initialized with invalid email $it")
                    }
                }
                val trackerId = connectifComponent.settingRepository.trackerId
                isInitialized = true
                connectifComponent.pushManager.pushStoredPushToken()
                Logger.i(
                    "Connectif initialized with apiKey $apiKey" +
                        if (!trackerId.isNullOrEmpty()) {
                            " and trackerId $trackerId"
                        } else {
                            " and no trackerId yet"
                        }
                )
            } catch (unauthorizedApiException: UnauthorizedApiException) {
                unauthorizedApiException.message?.let { Logger.e(it) }
            } catch (e: Exception) {
                Logger.e("Unable to initialize Connectif SDK: ${e.message}")
            }
        }
    }

    override fun addPushToken(token: String, context: Context) {
        runSequentially {
            try {
                if (isInitialized && (
                        !connectifComponent.settingRepository.webViewMode ||
                            !connectifComponent.settingRepository.trackerId.isNullOrEmpty()
                        )
                ) {
                    connectifComponent.pushManager.addPushToken(token)
                } else {
                    saveToLocalPushToken(token, context)
                    Logger.i(
                        "Push token saved locally, we will send it when Connectif SDK starts up"
                    )
                }
            } catch (unauthorizedApiException: UnauthorizedApiException) {
                unauthorizedApiException.message?.let { Logger.e(it) }
            } catch (exception: Exception) {
                Logger.e("An error occurred while trying to add the token")
            }
        }
    }

    internal fun saveToLocalPushToken(token: String, context: Context) {
        try {
            val sharedPreferences: SharedPreferences =
                context.getSharedPreferences(
                    Constants.Global.PREFERENCE_NAME,
                    Context.MODE_PRIVATE
                )
            val localPushTokenDataSource = LocalPushTokenDataSourceImpl(sharedPreferences)
            localPushTokenDataSource.pushToken = token
        } catch (exception: Exception) {
            Logger.e("An error occurred while trying to save the token locally")
        }
    }

    override fun handlePushNotification(data: Map<String, String>, context: Context): Boolean {
        if (!isInitialized) {
            Logger.e("Connectif is not initialized")
            return false
        }
        try {
            if (isConnectifPushNotification(data)) {
                val notificationData = data["data"]
                    ?: throw IllegalArgumentException(
                        "Missing 'data' key in the push notification data"
                    )
                connectifComponent.pushManager.handleConnectifPushNotification(
                    connectifComponent.pushManager.parseToNotificationData(notificationData),
                    context
                )
                return true
            }
        } catch (e: IllegalArgumentException) {
            Logger.e("Arguments error: ${e.message}")
        } catch (e: Exception) {
            Logger.e("Error handling push notification: ${e.message}")
        }
        return false
    }

    override fun isConnectifPushNotification(data: Map<String, String>?): Boolean {
        Logger.d(data.toString())
        return data?.containsKey(Constants.PushNotification.CN_PUSH_IDENTIFY) ?: false
    }

    private fun executeEvent(eventAction: suspend () -> Unit, callbacks: EventCallbacks?) {
        runSequentially {
            try {
                if (!isInitialized) {
                    logAndReportError("Connectif is not initialized", callbacks)
                    return@runSequentially
                }
                if (connectifComponent.settingRepository.webViewMode) {
                    logAndReportError(
                        "Error when trying to send an event, not allowed with WebView mode enabled",
                        callbacks
                    )
                    return@runSequentially
                }
                eventAction()
                reportSuccess(callbacks)
            } catch (validationException: ValidationException) {
                logAndReportError(validationException.messageErrors.toString(), callbacks)
            } catch (unauthorizedApiException: UnauthorizedApiException) {
                logAndReportError(unauthorizedApiException.message, callbacks)
            } catch (connectifApiException: ConnectifApiException) {
                logAndReportError(connectifApiException.message, callbacks)
            } catch (illegalArgumentException: IllegalArgumentException) {
                logAndReportError(illegalArgumentException.message, callbacks)
            } catch (exception: Exception) {
                exception.printStackTrace()
                logAndReportError(exception.message, callbacks)
            }
        }
    }

    override fun sendPageVisit(name: String, callbacks: EventCallbacks?) {
        executeEvent({
            connectifComponent.eventManager.sendPageVisit(PageVisit(name))
        }, callbacks)
    }

    override fun sendSearch(searchText: String, callbacks: EventCallbacks?) {
        executeEvent({
            connectifComponent.eventManager.sendSearch(Search(searchText))
        }, callbacks)
    }

    override fun sendProductVisit(product: Product, callbacks: EventCallbacks?) {
        executeEvent({
            connectifComponent.eventManager.sendProductVisit(ProductVisit(product))
        }, callbacks)
    }

    override fun sendLogin(email: String, callbacks: EventCallbacks?) {
        executeEvent({
            connectifComponent.eventManager.sendLogin(Login(email))
        }, callbacks)
    }

    override fun sendRegister(
        email: String,
        contactInfo: RegisterContactInfo?,
        callbacks: EventCallbacks?
    ) {
        executeEvent({
            connectifComponent.eventManager.sendRegister(Register(email, contactInfo))
        }, callbacks)
    }

    override fun sendContactInfo(contactInfo: ContactInfo, callbacks: EventCallbacks?) {
        executeEvent({
            connectifComponent.eventManager.sendContactInfo(contactInfo)
        }, callbacks)
    }

    override fun subscribeToNewsletter(callbacks: EventCallbacks?) {
        executeEvent({
            connectifComponent.eventManager.subscribeToNewsletter()
        }, callbacks)
    }

    override fun sendCustomEventById(
        eventId: String,
        customEvent: CustomEvent,
        callbacks: EventCallbacks?
    ) {
        executeEvent({
            connectifComponent.eventManager.sendCustomEventById(
                eventId = eventId,
                event = customEvent
            )
        }, callbacks)
    }

    override fun sendCustomEventById(eventId: String, payload: Any?, callbacks: EventCallbacks?) {
        executeEvent({
            connectifComponent.eventManager.sendCustomEventById(
                eventId,
                payload ?: CustomEvent()
            )
        }, callbacks)
    }

    override fun sendCustomEventByAlias(
        eventAlias: String,
        customEvent: CustomEvent,
        callbacks: EventCallbacks?
    ) {
        executeEvent({
            connectifComponent.eventManager.sendCustomEventByAlias(
                eventAlias = eventAlias,
                event = customEvent
            )
        }, callbacks)
    }

    override fun sendCustomEventByAlias(
        eventAlias: String,
        payload: Any?,
        callbacks: EventCallbacks?
    ) {
        executeEvent({
            connectifComponent.eventManager.sendCustomEventByAlias(
                eventAlias,
                payload ?: CustomEvent()
            )
        }, callbacks)
    }

    override fun sendCart(cart: Cart, callbacks: EventCallbacks?) {
        executeEvent({
            connectifComponent.eventManager.sendCart(
                cart
            )
        }, callbacks)
    }

    override fun sendPurchase(purchase: Purchase, callbacks: EventCallbacks?) {
        executeEvent({
            connectifComponent.eventManager.sendPurchase(
                purchase
            )
        }, callbacks)
    }

    internal fun trackPushClick(trackPushClickData: TrackPushClickData?) {
        runSequentially {
            if (!isInitialized) {
                Logger.e("Connectif is not initialized")
                return@runSequentially
            }

            trackPushClickData?.let {
                connectifComponent.pushManager.trackPushClick(
                    trackPushClickData
                )
            }
                ?: Logger.e("TrackPushClickData cant be null for track click")
        }
    }

    internal fun setTrackerId(trackedId: String) {
        runSequentially {
            try {
                with(connectifComponent) {
                    val oldTrackerId = settingRepository.trackerId
                    settingRepository.trackerId = trackedId

                    if (oldTrackerId != trackedId) {
                        localPushTokenDataSource.pushToken?.let {
                            localPushTokenDataSource.lastUpdate = Date(0)
                            pushManager.addPushToken(it)
                        }
                    }
                }
            } catch (e: Exception) {
                Logger.e("Error setting trackerId: ${e.message}")
            }
        }
    }

    private suspend fun logAndReportError(message: String?, callbacks: EventCallbacks?) {
        withContext(Dispatchers.Main) {
            callbacks?.onError(message)
        }
        message?.let { Logger.e(it) }
    }

    private suspend fun reportSuccess(callbacks: EventCallbacks?) {
        withContext(Dispatchers.Main) {
            callbacks?.onSuccess()
        }
    }
}
