package adfactory.player.android;

import android.content.Context;
import android.media.MediaPlayer;
import android.util.AttributeSet;
import android.widget.VideoView;

import java.util.ArrayList;
import java.util.List;

import adfactory.player.android.interfaces.IVideoPlayer;

/**
 * Created by medyo on 8/22/16.
 */
public class AdFactoryVideoView extends VideoView implements IVideoPlayer{

    int mVideoViewHeight;

    private final List<IPlayerCallback> mVideoPlayerCallbacks = new ArrayList<>(1);
    private PlaybackState mPlaybackState;


    private enum PlaybackState {
        STOPPED, PAUSED, PLAYING
    }

    public AdFactoryVideoView(Context context) {
        super(context);
        initialize();
    }

    public AdFactoryVideoView(Context context, AttributeSet attrs) {
        super(context, attrs);
        initialize();
    }

    private void initialize() {

        mVideoViewHeight = getResources().getDimensionPixelSize(R.dimen.adfactory_video_height);

        mPlaybackState = PlaybackState.STOPPED;

        super.setOnCompletionListener(new MediaPlayer.OnCompletionListener() {
            @Override
            public void onCompletion(MediaPlayer mediaPlayer) {
                mPlaybackState = PlaybackState.STOPPED;

                for (IPlayerCallback callback : mVideoPlayerCallbacks) {
                    callback.onCompleted();
                }
            }
        });

        super.setOnErrorListener(new MediaPlayer.OnErrorListener() {
            @Override
            public boolean onError(MediaPlayer mediaPlayer, int what, int extra) {
                mPlaybackState = PlaybackState.STOPPED;

                for (IPlayerCallback callback : mVideoPlayerCallbacks) {
                    callback.onError(mediaPlayer, what, extra);
                }
                return true;
            }
        });

        super.setOnPreparedListener(new MediaPlayer.OnPreparedListener() {
            @Override
            public void onPrepared(MediaPlayer mediaPlayer) {
                for (IPlayerCallback callback : mVideoPlayerCallbacks) {
                    callback.onPrepare();
                }
            }
        });

    }

    @Override
    public void start() {
        super.start();
        switch (mPlaybackState) {
            case STOPPED:
                for (IPlayerCallback callback : mVideoPlayerCallbacks) {
                    callback.onPlay();
                }
                break;
            case PAUSED:
                for (IPlayerCallback callback : mVideoPlayerCallbacks) {
                    callback.onResume();
                }
                break;
            default:
                // Already playing; do nothing.
        }
        mPlaybackState = PlaybackState.PLAYING;
    }

    @Override
    public void play() {
        start();
    }

    @Override
    public void stopPlayback() {
        super.stopPlayback();
        mPlaybackState = PlaybackState.STOPPED;
    }

    @Override
    public void addPlayerCallback(IPlayerCallback callback) {
        mVideoPlayerCallbacks.add(callback);
    }

    @Override
    public void setOnCompletionListener(MediaPlayer.OnCompletionListener listener) {
        // The OnCompletionListener can only be implemented by SampleVideoPlayer.
        throw new UnsupportedOperationException();
    }

    @Override
    public void setOnErrorListener(MediaPlayer.OnErrorListener listener) {
        // The OnErrorListener can only be implemented by SampleVideoPlayer.
        throw new UnsupportedOperationException();
    }

    @Override
    public void setOnPreparedListener(MediaPlayer.OnPreparedListener listener) {
        // The setOnPreparedListener can only be implemented by SampleVideoPlayer.
        throw new UnsupportedOperationException();
    }

    @Override
    protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
        int width = getDefaultSize(0, widthMeasureSpec);
        int height = getDefaultSize(0, heightMeasureSpec);

        setMeasuredDimension(width, height);
    }


}
