package adfactory.player.android;

import android.content.Context;
import android.content.res.Configuration;
import android.support.v4.content.ContextCompat;
import android.support.v7.app.AppCompatActivity;
import android.text.TextUtils;
import android.util.AttributeSet;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.view.ViewStub;
import android.view.Window;
import android.view.WindowManager;
import android.webkit.URLUtil;
import android.widget.RelativeLayout;
import android.widget.TextView;

import adfactory.player.android.Models.VideoModel;
import adfactory.player.android.Utils.Html5WebVideoView;
import adfactory.player.android.Utils.Utils;
import adfactory.player.android.Utils.VideoRequestTask;

/**
 * Created by medyo on 8/23/16.
 */
public class AdFactoryPlayer extends RelativeLayout {

    RelativeLayout mWrapper;
    TextView mTextError;

    AdFactoryVideoView mVideoView;
    View mRootView;
    VideoLoader mVideoLoader;
    int mVideoViewBackgroundColor;
    AdFactoryPlayerControllers mControllers;
    AdFactory mAdFactory;
    String mVideoUrl, mVideoId;
    onVideoReady mCallback;
    Boolean mSetupReady = false;
    Html5WebVideoView mWebView;
    String mApiKey;
    int mSeekBarColor;
    VideoModel mVideoModel;

    public interface onVideoReady{
        void onPlay();
        void onComplete();
        void onError();
        void onPause();
        void onResume();
        void onOrientationChange(boolean fullscreen);
    }

    public AdFactoryPlayer(Context context) {
        super(context);
        initView();
    }

    public AdFactoryPlayer(Context context, AttributeSet attrs) {
        super(context, attrs);
        initView();
    }

    public void setUrl(String url){
        this.mVideoUrl = url;
    }

    public void initView(){

        // Inflate Layout
        mRootView = LayoutInflater.from(getContext()).inflate(R.layout.adfactory_player, this);

        // Set View Background
        mVideoViewBackgroundColor = ContextCompat.getColor(getContext(), R.color.adfactory_default_background_color);
        mWrapper = (RelativeLayout) mRootView.findViewById(R.id.wrapper);
        mWrapper.setBackgroundColor(mVideoViewBackgroundColor);

        mSeekBarColor = Utils.getThemeAccentColor(getContext());
        mTextError = (TextView) mRootView.findViewById(R.id.error_text_view);

        mApiKey = Utils.getApiKey(getContext());
        mSetupReady = !TextUtils.isEmpty(mApiKey);

        if (!mSetupReady){
            Log.e("AdFactory", "Please add <meta-data android:name=\"com.adfactory.key\"\n" +
                    "            android:value=\"Your Api Key\" /> To your manifest file");
            displayError(getContext().getResources().getString(R.string.adfactory_exception_missing_key));
        } else {
            mTextError.setVisibility(GONE);
        }

    }

    public void setCallback(onVideoReady callback){
        this.mCallback = callback;
    }

    private void displayError(String error){
        mTextError.setText(error);
        mTextError.setVisibility(VISIBLE);
    }

    public void play (String embed){
        this.mVideoUrl = embed;

        if (mSetupReady){

            if (mVideoModel != null){
                return;
            }
            if (URLUtil.isHttpUrl(embed) || URLUtil.isHttpsUrl(embed)){
                this.mVideoId = Utils.extractVideoId(embed);
            } else {
                this.mVideoUrl = Utils.extractVideoUrl(embed);
                Log.e("Url", mVideoUrl);

                this.mVideoId = Utils.extractVideoId(mVideoUrl);
                Log.e("mVideoId", mVideoId);
            }

            if (TextUtils.isEmpty(mVideoId)){
                displayError(getResources().getString(R.string.adfactory_exception_video_not_found));
            } else {
                // We'll check video status to see if it's ready to display native player or a web view
                new VideoRequestTask(mVideoId, mApiKey, new VideoRequestTask.onVideoReady() {
                    @Override
                    public void onComplete(VideoModel video) {
                        mVideoModel = video;
                        if (video.getReady()){

                            // Inflate Video Player + Controllers
                            ViewStub videoPlayerStub = (ViewStub) mRootView.findViewById(R.id.stub_video_player);
                            videoPlayerStub.inflate();
                            mVideoView = (AdFactoryVideoView) mRootView.findViewById(R.id.adfactory_video_view);
                            mControllers = (AdFactoryPlayerControllers) mRootView.findViewById(R.id.adfactory_controllers);
                            mAdFactory = new AdFactory(mVideoView, mWrapper, video.getVideoSd());
                            mControllers.setup(mVideoView, mAdFactory, mSeekBarColor, mCallback);
                            mControllers.initialize();
                            mVideoLoader = new VideoLoader(getContext(), mAdFactory, mControllers, video.getDfp());
                            mVideoLoader.setup();

                            mVideoLoader.play();
                        } else {

                            // Inflate WebView
                            ViewStub videoWebViewStub = (ViewStub) mRootView.findViewById(R.id.stub_video_web_view);
                            videoWebViewStub.inflate();
                            mWebView = (Html5WebVideoView) mRootView.findViewById(R.id.adfactory_webView);
                            mWebView.setAutoPlay(true);
                            mWebView.loadDataWithBaseURL("", video.getEmbed().replace("https", "http"), "text/html", "UTF-8", "");
                        }

                    }

                    @Override
                    public void onError() {
                        displayError(getResources().getString(R.string.adfactory_exception_video_not_found));
                    }
                }).execute();
            }
        }



    }

    public void resume() {
        if (mVideoLoader != null){
            mVideoLoader.resume();
        }
    }

    public void pause() {
        if (mVideoLoader != null){
            mVideoLoader.pause();
        }
    }

    @Override
    protected void onConfigurationChanged(Configuration newConfig) {
        super.onConfigurationChanged(newConfig);
        handleOrientation();
    }

    protected void handleOrientation() {

        int orientation = getResources().getConfiguration().orientation;
        boolean fullscreen = (orientation == Configuration.ORIENTATION_LANDSCAPE);


        int videoHeight = getResources().getDimensionPixelSize(R.dimen.adfactory_video_height);

        if (fullscreen){
            getWindow().setFlags(WindowManager.LayoutParams.FLAG_FULLSCREEN,  WindowManager.LayoutParams.FLAG_FULLSCREEN);
            getWindow().clearFlags(WindowManager.LayoutParams.FLAG_FORCE_NOT_FULLSCREEN);
            RelativeLayout.LayoutParams params = (LayoutParams) mWrapper.getLayoutParams();
            params.width = LayoutParams.MATCH_PARENT;
            params.height = LayoutParams.MATCH_PARENT;
            mWrapper.setLayoutParams(params);
            if (mCallback != null){
                mCallback.onOrientationChange(true);
            }


        }else{
            getWindow().clearFlags(WindowManager.LayoutParams.FLAG_FULLSCREEN);
            getWindow().addFlags(WindowManager.LayoutParams.FLAG_FORCE_NOT_FULLSCREEN);
            RelativeLayout.LayoutParams videoItemParams = new RelativeLayout.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, videoHeight);
            mWrapper.setLayoutParams(videoItemParams);
            if (mCallback != null){
                mCallback.onOrientationChange(false);
            }
        }

        getWindow().getDecorView().requestLayout();
    }

    private Window getWindow(){
        return ((AppCompatActivity) getContext()).getWindow();
    }

    public AdFactoryPlayerControllers getControllers(){
        return mControllers;
    }

    public void setSeekBarColor(int color){
        mSeekBarColor = color;
    }
}
