package adfactory.player.android;

import android.content.Context;
import android.text.TextUtils;
import android.util.Log;

import com.google.ads.interactivemedia.v3.api.AdDisplayContainer;
import com.google.ads.interactivemedia.v3.api.AdErrorEvent;
import com.google.ads.interactivemedia.v3.api.AdEvent;
import com.google.ads.interactivemedia.v3.api.AdsLoader;
import com.google.ads.interactivemedia.v3.api.AdsManager;
import com.google.ads.interactivemedia.v3.api.AdsManagerLoadedEvent;
import com.google.ads.interactivemedia.v3.api.AdsRequest;
import com.google.ads.interactivemedia.v3.api.ImaSdkFactory;


/**
 * Created by medyo on 8/22/16.
 */
public class VideoLoader implements AdsLoader.AdsLoadedListener, AdErrorEvent.AdErrorListener, AdEvent.AdEventListener, AdFactory.OnContentCompleteListener, AdFactory.OnContentPreparedListener {

    private static final String TAG = "adFactory";
    String mVideoUrl, mDfpUrl;
    Context mContext;
    AdFactoryPlayerControllers mControllers;
    AdFactory mAdFactory;
    private AdsLoader mAdsLoader;

    // AdsManager exposes methods to control ad playback and listen to ad events.
    private AdsManager mAdsManager;

    private ImaSdkFactory mSdkFactory;

    private AdDisplayContainer mAdDisplayContainer;


    public VideoLoader(Context context, AdFactory adFactory, AdFactoryPlayerControllers controllers, String dfpUrl){
        Log.e(TAG, "init AdFactory Player ...");
        this.mAdFactory = adFactory;
        this.mContext = context;
        this.mControllers = controllers;
        this.mDfpUrl = dfpUrl;
        init();
    }

    private void init(){
        if (TextUtils.isEmpty(mVideoUrl)){
            Log.e(TAG, mContext.getString(R.string.adfactory_exception_invalid_video_url));
        }
    }

    public void setup() {

        mAdFactory.initialize();
        mAdFactory.setOnContentCompleteListener(this);
        mAdFactory.setOnContentPreparedListener(this);

        // Create an AdsLoader.
        mSdkFactory = ImaSdkFactory.getInstance();
        mAdsLoader = mSdkFactory.createAdsLoader(mContext);
        mAdsLoader.addAdErrorListener(this);
        mAdsLoader.addAdsLoadedListener(this);
    }

    @Override
    public void onAdsManagerLoaded(AdsManagerLoadedEvent adsManagerLoadedEvent) {
        Log.d(TAG, "onAdsManagerLoaded");
        mAdsManager = adsManagerLoadedEvent.getAdsManager();

        // Attach event and error event listeners.
        mAdsManager.addAdErrorListener(this);
        mAdsManager.addAdEventListener(this);
        mAdsManager.init();

    }

    @Override
    public void onAdError(AdErrorEvent adErrorEvent) {
        Log.e(TAG, "Ad Error: " + adErrorEvent.getError().getMessage());
        mAdFactory.resumeContentAfterAdPlayback();
    }

    @Override
    public void onAdEvent(AdEvent adEvent) {
        Log.e(TAG, "Event: " + adEvent.getType());

        // These are the suggested event types to handle. For full list of all ad event types,
        // see the documentation for AdEvent.AdEventType.
        switch (adEvent.getType()) {
            case LOADED:
                // AdEventType.LOADED will be fired when ads are ready to be played.
                // AdsManager.start() begins ad playback. This method is ignored for VMAP or ad
                // rules playlists, as the SDK will automatically start executing the playlist.
                mAdsManager.start();
                break;
            case CONTENT_PAUSE_REQUESTED:
                // AdEventType.CONTENT_PAUSE_REQUESTED is fired immediately before a video ad is
                // played.
                mAdFactory.pauseContentForAdPlayback();
                break;
            case CONTENT_RESUME_REQUESTED:
                // AdEventType.CONTENT_RESUME_REQUESTED is fired when the ad is completed and you
                // should start playing your content.
                mAdFactory.resumeContentAfterAdPlayback();
                break;
            case ALL_ADS_COMPLETED:
                if (mAdsManager != null) {
                    mAdsManager.destroy();
                    mAdsManager = null;
                }
                break;
            default:
                break;
        }
    }

    private void requestAds() {
        requestAds(mDfpUrl);
    }

    private void requestAds(String adTagUrl) {
        Log.d(TAG, "requestAds");
        mAdDisplayContainer = mSdkFactory.createAdDisplayContainer();
        mAdDisplayContainer.setPlayer(mAdFactory.getVideoAdPlayer());
        mAdDisplayContainer.setAdContainer(mAdFactory.getAdUiContainer());

        // Create the ads request.
        AdsRequest request = mSdkFactory.createAdsRequest();
        request.setAdTagUrl(adTagUrl);
        request.setAdDisplayContainer(mAdDisplayContainer);
        request.setContentProgressProvider(mAdFactory.getContentProgressProvider());

        // Request the ad. After the ad is loaded, onAdsManagerLoaded() will be called.
        mAdsLoader.requestAds(request);
    }

    public void play() {
        requestAds();
    }

    /**
     * Resumes video playback.
     */
    public void resume() {
        Log.d(TAG, "resume");
        mAdFactory.restorePosition();
        if (mAdsManager != null && mAdFactory.getIsAdDisplayed()) {
            mAdsManager.resume();
        }
    }

    /**
     * Pauses video playback.
     */
    public void pause() {
        Log.d(TAG, "pause");
        mAdFactory.savePosition();
        if (mAdsManager != null && mAdFactory.getIsAdDisplayed()) {
            mAdsManager.pause();
        }
    }

    @Override
    public void onContentComplete() {
        mAdsLoader.contentComplete();
    }

    @Override
    public void onContentPrepared() {
        Log.d("VideoLoader", "onContentPrepared");
    }
}
