package adfactory.player.android;

import android.media.MediaPlayer;
import android.net.Uri;
import android.os.Build;
import android.text.TextUtils;
import android.util.Log;
import android.view.ViewGroup;
import android.widget.VideoView;

import com.google.ads.interactivemedia.v3.api.player.ContentProgressProvider;
import com.google.ads.interactivemedia.v3.api.player.VideoAdPlayer;
import com.google.ads.interactivemedia.v3.api.player.VideoProgressUpdate;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import adfactory.player.android.interfaces.IVideoPlayer;

/**
 * Created by medyo on 8/22/16.
 */
public class AdFactory {


    public interface OnContentCompleteListener {
        void onContentComplete();
    }

    public interface OnContentPreparedListener {
        void onContentPrepared();
    }

    // Video View
    private AdFactoryVideoView mAdFactoryVideoView;

    private VideoAdPlayer mVideoAdPlayer;

    // Used to track if the current video is an ad (as opposed to a content video).
    private boolean mIsAdDisplayed;

    // Used to track the current content video URL to resume content playback.
    private String mContentVideoUrl;

    // The saved position in the content to resume to after ad playback.
    private int mSavedVideoPosition;

    // Check if the content video is complete
    private boolean mIsContentComplete;

    // ContentProgressProvider interface implementation for the SDK to check content progress.
    private ContentProgressProvider mContentProgressProvider;

    // Called when the content is completed.
    private OnContentCompleteListener mOnContentCompleteListener;

    // Called when the content is completed.
    private OnContentPreparedListener mOnContentPreparedListener;

    private final List<VideoAdPlayer.VideoAdPlayerCallback> mAdCallbacks =
            new ArrayList<VideoAdPlayer.VideoAdPlayerCallback>(1);


    // The SDK will render ad playback UI elements into this ViewGroup.
    private ViewGroup mAdUiContainer;

    public AdFactory(AdFactoryVideoView videoPlayer, ViewGroup adUIContainer, String url) {
        mAdFactoryVideoView = videoPlayer;
        mAdUiContainer = adUIContainer;
        mContentVideoUrl = url;
    }


    public void initialize() {
        mIsAdDisplayed = false;
        mSavedVideoPosition = 0;
        mIsContentComplete = false;

        Log.e("AdFactory.java", "initialize");

        mVideoAdPlayer = new VideoAdPlayer() {

            @Override
            public VideoProgressUpdate getAdProgress() {
                if (!mIsAdDisplayed || mAdFactoryVideoView.getDuration() <= 0) {
                    return VideoProgressUpdate.VIDEO_TIME_NOT_READY;
                }
                return new VideoProgressUpdate(mAdFactoryVideoView.getCurrentPosition(),mAdFactoryVideoView.getDuration());
            }

            @Override
            public void playAd() {
                mIsAdDisplayed = true;
                mAdFactoryVideoView.play();
            }

            @Override
            public void loadAd(String url) {
                Log.e("AdFactory.java", "loadAd");
                mIsAdDisplayed = true;
                mAdFactoryVideoView.setVideoPath(url);
            }

            @Override
            public void stopAd() {
                mAdFactoryVideoView.stopPlayback();
            }

            @Override
            public void pauseAd() {
                mAdFactoryVideoView.pause();
            }

            @Override
            public void resumeAd() {
                playAd();
            }

            @Override
            public void addCallback(VideoAdPlayerCallback videoAdPlayerCallback) {
                mAdCallbacks.add(videoAdPlayerCallback);
            }

            @Override
            public void removeCallback(VideoAdPlayerCallback videoAdPlayerCallback) {
                mAdCallbacks.remove(videoAdPlayerCallback);
            }
        };

        mContentProgressProvider = new ContentProgressProvider() {
            @Override
            public VideoProgressUpdate getContentProgress() {
                if (mIsAdDisplayed || mAdFactoryVideoView.getDuration() <= 0) {
                    return VideoProgressUpdate.VIDEO_TIME_NOT_READY;
                }
                return new VideoProgressUpdate(mAdFactoryVideoView.getCurrentPosition(),mAdFactoryVideoView.getDuration());
            }
        };

        mAdFactoryVideoView.addPlayerCallback(new IVideoPlayer.IPlayerCallback() {
            @Override
            public void onPlay() {
                if (mIsAdDisplayed) {
                    Log.d("AdFactory", "onPlay");
                    for (VideoAdPlayer.VideoAdPlayerCallback callback : mAdCallbacks) {
                        callback.onPlay();
                    }
                }
            }

            @Override
            public void onCompleted() {
                Log.d("AdFactory", "onCompleted");
                if (mIsAdDisplayed) {
                    for (VideoAdPlayer.VideoAdPlayerCallback callback : mAdCallbacks) {
                        callback.onEnded();
                    }
                } else {

                    if (mOnContentCompleteListener != null) {
                        mOnContentCompleteListener.onContentComplete();
                    }
                    mIsContentComplete = true;
                }
            }

            @Override
            public void onError(MediaPlayer mp, int what, int extra) {
                Log.e("AdFactory.java", "onError");
                if (mIsAdDisplayed) {
                    for (VideoAdPlayer.VideoAdPlayerCallback callback : mAdCallbacks) {
                        callback.onError();
                    }
                }
            }

            @Override
            public void onPause() {
                if (mIsAdDisplayed) {
                    Log.d("AdFactory", "onPause");
                    for (VideoAdPlayer.VideoAdPlayerCallback callback : mAdCallbacks) {
                        callback.onPause();
                    }
                }
            }

            @Override
            public void onResume() {
                if (mIsAdDisplayed) {
                    Log.e("AdFactory.java", "onResume");
                    for (VideoAdPlayer.VideoAdPlayerCallback callback : mAdCallbacks) {
                        callback.onResume();
                    }
                }
            }

            @Override
            public void onPrepare() {
                Log.e("AdFactory.java", "onPrepare");

                if (!mIsAdDisplayed) {
                    if (mOnContentPreparedListener != null) {
                        mOnContentPreparedListener.onContentPrepared();
                    }
                    mIsContentComplete = true;
                }
            }
        });
    }

    public void savePosition() {
        mSavedVideoPosition = mAdFactoryVideoView.getCurrentPosition();
    }

    public void pauseContentForAdPlayback() {
        savePosition();
        mAdFactoryVideoView.stopPlayback();
    }

    public void restorePosition() {
        mAdFactoryVideoView.seekTo(mSavedVideoPosition);
    }

    public ViewGroup getAdUiContainer() {
        return mAdUiContainer;
    }

    public boolean getIsAdDisplayed() {
        return mIsAdDisplayed;
    }

    public ContentProgressProvider getContentProgressProvider() {
        return mContentProgressProvider;
    }

    public VideoAdPlayer getVideoAdPlayer() {
        return mVideoAdPlayer;
    }

    public void setOnContentCompleteListener(OnContentCompleteListener listener) {
        mOnContentCompleteListener = listener;
    }

    public void setOnContentPreparedListener(OnContentPreparedListener listener) {
        mOnContentPreparedListener = listener;
    }

    public void resumeContentAfterAdPlayback() {

        mIsAdDisplayed = false;

        if (!TextUtils.isEmpty(mContentVideoUrl)){
            Map<String, String> map = new HashMap<>();
            map.put("Referer", mContentVideoUrl);
            map.put("app-source", BuildConfig.APPLICATION_ID);
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                mAdFactoryVideoView.setVideoURI(Uri.parse(mContentVideoUrl), map);
            } else {
                mAdFactoryVideoView.setVideoURI(Uri.parse(mContentVideoUrl));
                try {
                    Field field = VideoView.class.getDeclaredField("mHeaders");
                    field.setAccessible(true);
                    field.set(mAdFactoryVideoView,  map);
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }else {
            mAdFactoryVideoView.setVideoPath(mContentVideoUrl);
        }

        mAdFactoryVideoView.setVideoPath(mContentVideoUrl);
        restorePosition();
        if (!mIsContentComplete) {
            mAdFactoryVideoView.play();
        } else {
            mAdFactoryVideoView.stopPlayback();
        }
    }

    public void play() {
        mIsContentComplete = false;
        mAdFactoryVideoView.setVideoPath(mContentVideoUrl);
        mAdFactoryVideoView.start();
    }

}
